/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GInterval                                                             *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <iostream>

#include "TString.h"

#include "Gfitter/GInterval.h"
#include "Gfitter/GStringList.h"

ClassImp(Gfitter::GInterval)

using namespace std;

Gfitter::GInterval::GInterval()
   : GObject(),
     m_min( 0 ), 
     m_max( 0 ),
     m_active( kFALSE )
{}

Gfitter::GInterval::GInterval( const TString& interval )
   : GObject(),
     m_min( 0 ), 
     m_max( 1 ),
     m_nsteps( 0 ),
     m_active( kFALSE )
{
   InitClassName( "GInterval" );

   // interpret string, which must be of the form of "[a,b]"   
   GStringList vlist( interval, ":" );
   if (!interval.Contains("[") || !interval.Contains("]") || !interval.Contains(":") 
       || (vlist.GetEntries() < 2 || vlist.GetEntries() > 3)) {
      m_logger << kFATAL << "Problem in constructing string: " << interval << GEndl;
   }
   TString a = vlist[0], b = vlist[1]; 

   a.ReplaceAll( "[", "" );  
   b.ReplaceAll( "]", "" );
   m_min = a.Atof(); 
   m_max = b.Atof(); 

   // sanity check
   if (m_min > m_max) m_logger << kFATAL << "min > max: " << m_min << " , " << m_max << GEndl;

   // set active if interval is non-trivial
   if (m_min < m_max) m_active = kTRUE;

   // check if nsteps are given, unless keep as default
   if (vlist.GetEntries() >= 3) {
      TString c = vlist[2]; c.ToLower();      
      if (!c.Contains("nsteps") || !c.Contains("=")) {
         m_logger << kFATAL << "Problem in constructing string. Unknown string format: " 
                  << interval << " (should be: \"[min:max]:nsteps=100\", where the nsteps are optional)" << GEndl;
      }
      GStringList wlist( c, "=" );
      if (wlist.GetEntries() != 2) {
         m_logger << kFATAL << "Problem in constructing string (2): " << interval << GEndl;
      }
      m_nsteps = wlist[1].Atoi();
   }      
   else m_nsteps = 0;
}

Gfitter::GInterval::GInterval( Double_t min, Double_t max, Int_t nsteps )
   : GObject(),
     m_min( min ), 
     m_max( max ),
     m_nsteps( nsteps ),
     m_active( kFALSE )
{
   InitClassName( "GInterval" );
   
   // sanity check
   if (m_min > m_max) m_logger << kFATAL << "min > max: " << m_min << " , " << m_max << GEndl;

   // set active if interval is non-trivial
   if (m_min < m_max) m_active = kTRUE;
}

Gfitter::GInterval::GInterval( const GInterval& interval )
   : GObject(),
     m_min   ( interval.GetMin() ), 
     m_max   ( interval.GetMax() ),
     m_nsteps( interval.GetNsteps() ),
     m_active( interval.IsActive() )
{
   InitClassName( "GInterval" );
   
   // sanity check
   if (m_min > m_min) m_logger << kFATAL << "min > max: " << m_min << " , " << m_max << GEndl ;
}

Gfitter::GInterval::~GInterval()
{}

Double_t Gfitter::GInterval::GetScanValue( Int_t step ) const
{
   if (step < 0 || step >= GetNsteps() || GetNsteps() <= 0) {
      m_logger << kFATAL << "<GetScanValue> Wrong step: " << step << " not in [0," << GetNsteps()-1 << "]" << GEndl;
   }
   
   return (step + 0.5)*GetWidth()/Float_t(GetNsteps()) + GetMin();
}

ostream& Gfitter::operator << ( ostream& os, const Gfitter::GInterval& interval ) 
{
   os << " [" << interval.GetMin() << ", " << interval.GetMax() << "]";
   if (interval.GetNsteps() > 0) os << " (nsteps: " << interval.GetNsteps() << ")";

   return os;
}
