/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GInterval                                                             *
 *                                                                                *
 * Description:                                                                   *
 *      Interval container                                                        *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef Gfitter_GInterval
#define Gfitter_GInterval

#include "Gfitter/GObject.h"

namespace Gfitter {

   class GInterval;
   ostream& operator << ( ostream& os, const GInterval& interval );

   class GInterval : public GObject {
    
   public:

      // default constructor (for persistency only - don't use)
      GInterval(); 

      GInterval( const TString& interval ); // string in form of "[a:b]"
      GInterval( Double_t min, Double_t max, Int_t nsteps = 0 );
      GInterval( const GInterval& ); // copy constructor (cloning input GData)
      ~GInterval();

      // accessors 
      Double_t GetMin()     const { return m_min; }
      Double_t GetMax()     const { return m_max; }
      Double_t GetWidth()   const { return m_max - m_min; }
      Double_t GetMean()    const { return 0.5*(m_min + m_max); }

      Bool_t   InInterval( Double_t v ) const { return !m_active || (v >= m_min && v <= m_max); }      
      Bool_t   InInterval  ( const GInterval& i ) const { 
         return InInterval( i.GetMin() ) && InInterval( i.GetMax() ); 
      }      
      Bool_t   IsActive()               const { return m_active; }

      Int_t    GetNsteps()    const { return m_nsteps; }
      Double_t GetScanValue( Int_t step ) const;

   private:

      Double_t m_min;
      Double_t m_max;
      Int_t    m_nsteps;   // an interval can be subdivided into steps (bins)
      Bool_t   m_active;   // is true if m_max > m_min (ie, it is a non-trivial interval)

      ClassDef(GInterval,1)
   };
}

#endif
