/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GMinuitFitter                                                         *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include "Riostream.h"
#include "TMinuit.h"
#include "Gfitter/GMinuitFitter.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GData.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GResultEval.h"
#include "Gfitter/GTimer.h"

using namespace std;

ClassImp(Gfitter::GMinuitFitter)

Gfitter::GMinuitFitter::GMinuitFitter() 
   : GFitterBase(),
     m_minuit( 0 ),
     m_useImprove( kFALSE ),
     m_useMinos( kFALSE ),
     m_isFirstInit( kTRUE ),
     m_fitCovMat( 0 )
{
   InitClassName( "GMinuitFitter" );
}

Gfitter::GMinuitFitter::~GMinuitFitter()
{
   if (m_minuit)    delete m_minuit;
   if (m_fitCovMat) delete [] m_fitCovMat;
}

void Gfitter::GMinuitFitter::Initialise()
{
   // run the prefitter first
   if (gStore()->GetPreFitter()) {
      gStore()->GetPreFitter()->Initialise();
      gStore()->GetPreFitter()->ExecuteFit();
   }

   m_This = this;      

   // base class call
   GFitterBase::Initialise();
   
   if( !m_isFirstInit )
      GetThisPtr()->SetFirstPassed();
   
   // delete TFitter instance if existing
   delete m_minuit; m_minuit = 0;
   
   // reset vectors
   GetFitPars().clear();
   GetPenalisedPars().clear();      
   
   // instantiate minuit
   // maximum number of fit parameters is equal to active GParameters
   // (2xnpar as workaround for TMinuit allocation bug (taken from RooMinuit))
   m_minuit = new TMinuit( 2*gStore()->GetNumOfActiveParametersWithoutTheory() );
   
   // minuit-specific settings
   Double_t args[10];

   // first thing: set output level      
   args[0] = (gStore()->ExistVariable( "MinuitFitter::PrintLevel" ) ? 
              gStore()->GetVariable( "MinuitFitter::PrintLevel" )->GetIntValue() : -1);
   this->checkErr( "SET PRINTOUT", ExecuteCommand( "SET PRINTOUT", args, 1 ) );

   // set fitter object
   GetMinuit()->SetObjectFit( this );
   GetMinuit()->Clear();
   
   // define fit parameters      
   Int_t npars = 0;
   GParPtrVec_t::const_iterator par = gStore()->GetActiveParameters().begin();
   for (; par != gStore()->GetActiveParameters().end(); ++par) {
      
      // is theoretical parameter (as opposed to measurement)
      // ... and is not actually a parameter that is scanned
      // (if it were scanned, it should enter the fit, but not be free to vary)
         
      // NOTE: it may occur that several measurements of a same GParameter "A"
      // were added to the datacard. If that parameter does not have a theory, 
      // we should add only once the parameter to the fit, and the corresponding 
      // A::m_fitValues need all to be set to this single free parameter
      
      // We implement this by adding a flag "IsFirst()" to each parameter. If they 
      // are unique, the flag is "T". If they are non-unique, the flag is only "T"
      // for the first parameter added to the GStore. The parameter will then hold 
      // a list of pointer to the other parameters of the same name, and update their
      // fit values  

      if ((*par)->VariesInFit()) {

         // save name and pointer to parameter that is free in the fit
         GetFitPars().push_back( *par );

         m_logger << kDEBUG << "Adding parameter to fit configuration : " << (*par)->GetParName() << GEndl;
         
         // sanity check
         if ((*par)->GetFitStep() <= 0) m_logger << kFATAL << "Huge troubles with fit step of par: " 
                                                 << (*par)->GetParName() << GEndl;
    
         m_logger << kINFO << (*par)->GetParName() 
                  << ", MinFitRange : " << (*par)->GetFitRange().GetMin()
                  << ", MaxFitRange : " << (*par)->GetFitRange().GetMax()
                  << GEndl;
                  
         // book the fit parameter
         SetParameter( npars++, 
                       (*par)->GetParName(), 
                       (*par)->GetFitValue(), 
                       (*par)->GetFitStep(),
                       (*par)->GetFitRange().GetMin(), 
                       (*par)->GetFitRange().GetMax() );
      }  
   }
   
   // if the parameter has a theory and should be scanned, we need to add 
   // a penalty function to the fit that guarantees that the theory always
   // corresponds to the wanted value in the scan   

   // JH: loop over all Parameters needed here! Scan!
   for (par = gStore()->GetParameters().begin(); par != gStore()->GetParameters().end(); ++par) {
      // change from MG: We do not need IsFirst(), don't we?
      if ((*par)->HasTheory() && (*par)->IsScanned() ) { 
         m_logger << kINFO << "Setting penalised Parameter: " << (*par)->GetParName() << GEndl;
         GetPenalisedPars().push_back( (*par) );
      }
   }        
   
   // set internally the number of fit parameters (needed due to TFitter "feature" of 
   // "current" free parameters given to FCN only)
   this->SetNumOfFitPars( npars );
   
   // define the FCN function interface
   GetMinuit()->SetFCN( &IFCN );
      
   // error level: 1 (2*log(L) fit
   args[0] = 1;
   ExecuteCommand( "SET ERR", args, 1 );
   
   // use improve and/or minos ?
   if (gStore()->ExistVariable( "MinuitFitter::UseImprove" )) {
      m_useImprove = gStore()->GetVariable( "MinuitFitter::UseImprove" )->GetBoolValue();
   }
   if (gStore()->ExistVariable( "MinuitFitter::UseMinos" )) {
      m_useMinos = gStore()->GetVariable( "MinuitFitter::UseMinos" )->GetBoolValue();
   }

   // print warnings ?
   if (gStore()->ExistVariable( "MinuitFitter::NoWarnings" )) {
      if (gStore()->GetVariable( "MinuitFitter::NoWarnings" )->GetBoolValue())
         this->checkErr( "SET NOWARNINGS", ExecuteCommand( "SET NOWARNINGS", args, 0 ) );
   }

   // define fit strategy
   if (gStore()->ExistVariable( "MinuitFitter::Strategy" )) {
      args[0] = gStore()->GetVariable( "MinuitFitter::Strategy" )->GetIntValue();
   }
   else args[0] = 2;

   this->checkErr( "SET STRATEGY", ExecuteCommand( "SET STRATEGY", args, 1 ) );
   
   // sanity check before computing correlation errors 
   if (!gStore()->IsCorrelationActive() && 
       gStore()->GetActiveParameters().size() != gStore()->GetActiveUncorrelatedParameters().size()) {
      m_logger << kFATAL << "<Initialise> (Correlations inactive) error in number of parameters: "
               << gStore()->GetActiveParameters().size() << " | " 
               << gStore()->GetActiveCorrelatedParameters().size() << " | " 
               << gStore()->GetActiveUncorrelatedParameters().size() << GEndl;
   }
   
   // init covariance matrices and/or diagonal errors
   if (gStore()->IsCorrelationActive()) this->InitErrorMatrix();

   // some debug output, only when function is called first
   if( m_isFirstInit ){
      PrintConfiguration();
      m_isFirstInit = kFALSE;
   }

}
   
Double_t Gfitter::GMinuitFitter::GetCurrentChi2()
{
   // update it parameters
   
   Double_t chi2, *gin( 0 ), *fitPars( 0 );
   Int_t    iflag( 0 ), npars( 0 );

   fitPars = new Double_t[GetNumOfFitPars()];
   for (Int_t ipar=0; ipar < GetNumOfFitPars(); ++ipar) fitPars[ipar] = GetFitPars()[ipar]->GetFitValue();

   this->IFCN( npars, gin, chi2, fitPars, iflag );

   delete [] fitPars; 
   return chi2;
}

Double_t Gfitter::GMinuitFitter::ExecuteFit()
{   
   // short cut if no free parameters: compute FCN right away
   if (GetFitPars().size() == 0) {
      Double_t chi2, *gin( 0 ), *fitPars( 0 );
      Int_t    iflag( 0 ), npars( 0 );
      this->IFCN( npars, gin, chi2, fitPars, iflag );
      return chi2;
   }   

   // init timer
   GTimer timer;
   
   // continue with usual case
   Double_t args[10];
    
   // apply pre-fit scan ?
   for (Int_t ipar=0; ipar < GetNumOfFitPars(); ++ipar) {
      const GInterval* pfs = GetFitPars()[ipar]->GetPreFitScanRange();
      if (pfs) {
         Double_t bestc2 = GetCurrentChi2();
         Double_t bestv  = GetFitPars()[ipar]->GetFitValue();
         for (Int_t i=0; i<pfs->GetNsteps(); i++) {
            GetFitPars()[ipar]->SetFitValue( pfs->GetScanValue(i) );
            Double_t c2 = GetCurrentChi2();
            if (c2 < bestc2) { bestc2 = c2; bestv = GetFitPars()[ipar]->GetFitValue(); }
         }
         GetFitPars()[ipar]->SetFitValue( bestv );
      }
   }

   if (true) {
//   if (false) {
      m_logger << kDEBUG << "=====================================" << GEndl;
      GParPtrVec_t::const_iterator par;
      const GParPtrVec_t& v = gStore()->GetActiveParameters();
      
      // debug output
      if (m_logger.GetMinLevel() <= kDEBUG) {
         for (par = v.begin(); par != v.end(); par++) {
            Double_t value;
            if (!(*par)->HasTheory()) value = (*par)->GetFitValue();
            else                      value = (*par)->GetTheoryPrediction();
            m_logger << kDEBUG << "Fit par " << (*par)->GetFullName()
                     << "\" initial value (v/fit): ("
                     << (*par)->GetValue() << "/" << value << ")"
                     << ", fit range: [" << (*par)->GetFitRange().GetMin() << ", "
                     << (*par)->GetFitRange().GetMax() << "]"
                     << ", fit step: " << (*par)->GetFitStep()
                     << ", chi2: " << (*par)->GetChiSquared()
                     << GEndl;
         }
      }
      m_logger << kDEBUG << "=====================================" << GEndl;
     
      /// reset fit parameters      
         Int_t npars = 0;
         par = gStore()->GetActiveParameters().begin();
         for (; par != gStore()->GetActiveParameters().end(); ++par) {

            if ((*par)->VariesInFit()) {

               /// save name and pointer to parameter that is free in the fit
               //GetFitPars().push_back( *par );

               /// sanity check
               if ((*par)->GetFitStep() <= 0) m_logger << kFATAL << "Huge troubles with fit step of par: "
                                                       << (*par)->GetParName() << GEndl;

               // book the fit parameter
               SetParameter( npars++,
                             (*par)->GetParName(),
                             (*par)->GetFitValue(),
                             (*par)->GetFitStep(),
                             (*par)->GetFitRange().GetMin(),
                             (*par)->GetFitRange().GetMax() );
            }
         }

         m_logger << kDEBUG << "=====================================" << GEndl;
   }

   // reset minimum chi2 --> this parameter is to check that the fitter really returns the 
   //                        minimum chi2 found
   m_chi2Min = 1e20;

   // minimize with MIGRAD
   args[0] = 10000*GetNumOfFitPars(); // maximum function calls
   args[1] = 0.1;                    // tolerance at minimum
   this->checkErr( "MIGrad",  ExecuteCommand( "MIGrad", args, 2 ), 0, -123, kTRUE );

   // improve minimum (returns 4 if no new minimum found -> this is NOT a problem !)
   if (m_useImprove) {
      args[0] = 10000*GetNumOfFitPars(); // maximum function calls
      ExecuteCommand( "IMProve", args, 1 );
   }
   
   // expensive MINOS analysis
   if (m_useMinos ) {
      args[0] = 5000;
      this->checkErr( "MINOs", ExecuteCommand( "MINOs", args, 1 ), 0, -123, kTRUE );
   }

   // measure time for fit
   m_fitDuration = timer.GetElapsedTime();
  
   // retrieve fit result
   Double_t chi2;
   Double_t edm;
   Double_t errdef; 
   Int_t    nvpar;
   Int_t    nparx;
   GetStats( chi2, edm, errdef, nvpar, nparx );

   // sanity check: has the minimum chi2 been returned ?
   static Int_t chi2_message_counter = 0;
   if (chi2 > m_chi2Min + 100*edm) {
      chi2_message_counter++;
      if (chi2_message_counter <= 10) {
         m_logger << kWARNING << "Fit did not return solution corresponding to minimum chi2. Fit returned: chi2 = "
                  << chi2 << "; minimum was: chi2 = " << m_chi2Min 
                  << " (EDM = " << edm << ")"
                  << " ==> correct chi2"
                  << GEndl;
      }
      if (chi2_message_counter == 10) {
         m_logger << kWARNING << "Suppress this message from now on" << GEndl;
      }
      chi2 = m_chi2Min;
   }

   // sanity check
   if (GetFitPars().size() != (UInt_t)nparx) {
      m_logger << kFATAL << "<ExecuteFit> Mismatch in number of parameters: "
               << GetFitPars().size() << " != " << nparx << GEndl;
   }
   
   this->SetEstimator( chi2 );
   this->SetEDM( edm );      
 
   for (UInt_t ipar=0; ipar<GetFitPars().size(); ++ipar) {
      Double_t val, errp, errm, errsym, globcor;
      val = GetParameter( ipar );
      GetErrors( ipar, errp, errm, errsym, globcor );

      // set parameter to fit result
      GetFitPars()[ipar]->SetFitValue( val );
   }
   
   return chi2;
}

void Gfitter::GMinuitFitter::DrawContour( GParameter* gpar1, GParameter* gpar2, 
                                          Int_t npoints, Double_t dchi2, Double_t* npx, Double_t* npy )
{
   // this is not very elegant... and even worse wrong if the active pars dont start at index 0
   Int_t par1=-999, par2=-999;
   for (Int_t ipar=0; ipar < GetNumOfFitPars(); ipar++) {
      if (GetFitPars()[ipar] == gpar1) par1 = ipar;
      if (GetFitPars()[ipar] == gpar2) par2 = ipar;
   }      
   if (par1 < 0 || par2 < 0) {
      m_logger << kFATAL << "Could not attribute contour parameters to fit parameters: "
               << gpar1->GetParName() << " " << gpar2->GetParName() << GEndl;
   }
         
   Int_t ierr;
   m_minuit->SetErrorDef( dchi2 );
   m_minuit->mncont( par1, par2, npoints, npx, npy, ierr );
}

void Gfitter::GMinuitFitter::UpdateResults()
{
   // the parameter vector
   const GParPtrVec_t& fp = GetFitPars();
   const UInt_t npar = fp.size();

   // retrieve covariance matrix first
   TMatrixD* covMat = new TMatrixD( npar, npar );
   for (UInt_t i=0; i < npar; i++) {
      for (UInt_t j=0; j < npar; j++) (*covMat)(i,j) = GetCovarianceMatrixElement(i,j);
   }
   
   for (UInt_t ipar=0; ipar < npar; ipar++) {

      Double_t val, errp, errm, errsym, globcor, chi2, edm, errdef;
		Int_t nvpar, nparx;

      val = GetParameter( ipar );
      GetErrors( ipar, errp, errm, errsym, globcor );
		GetStats ( chi2, edm, errdef, nvpar, nparx );

      // NOTE: covariance matrix is now owned by GResultEval: all objects have the same pointer reference
      fp[ipar]->SetResult( new GResultEval( val, chi2, errp, errm, errsym, globcor, covMat, GetFitDuration() ) );
	}
}

void Gfitter::GMinuitFitter::IFCN( Int_t& /* npars */, Double_t* /* grad */, Double_t &f, Double_t* fitPars, Int_t iflag )
{
   // Evaluate the minimisation function ----------------------------------------------------
   //
   //  Input parameters:
   //    npars:   number of currently variable parameters
   //             CAUTION: this is not (necessarily) the dimension of the fitPars vector !
   //    fitPars: array of (constant and variable) parameters
   //    iflag:   indicates what is to be calculated (see example below)
   //    grad:    array of gradients
   //
   //  Output parameters:
   //    f:       the calculated function value.
   //    grad:    the (optional) vector of first derivatives).
   // ---------------------------------------------------------------------------------------
   (GetThisPtr())->FCN( f, fitPars, iflag );
}

Gfitter::GMinuitFitter* Gfitter::GMinuitFitter::m_This = 0;

Gfitter::GMinuitFitter* Gfitter::GMinuitFitter::GetThisPtr()
{
   return m_This;
}

// ------------------------------------------------------------------------------------------
// T M i n u i t   W r a p p e r s   
// ------------------------------------------------------------------------------------------

Int_t Gfitter::GMinuitFitter::ExecuteCommand( const char *command, Double_t *args, Int_t nargs )
{
   Int_t ierr = 0;
   GetMinuit()->mnexcm( command, args, nargs, ierr );
   //m_logger << kINFO << command << " <EXECUTE> " << ierr << GEndl; 
   return ierr;
}

Int_t Gfitter::GMinuitFitter::GetStats( Double_t &amin, Double_t &edm, Double_t &errdef, 
                                        Int_t &nvpar, Int_t &nparx ) const 
{
   Int_t ierr = 0;
   GetMinuit()->mnstat( amin, edm, errdef, nvpar, nparx, ierr );
   return ierr;
}

Double_t Gfitter::GMinuitFitter::GetParameter( Int_t ipar ) const
{
   // return current value of parameter ipar
   Int_t ierr = 0;
   TString pname;
   Double_t value,verr,vlow,vhigh;
   
   GetMinuit()->mnpout( ipar, pname, value, verr, vlow, vhigh, ierr );
   return value;   
}

TString Gfitter::GMinuitFitter::GetParName2( Int_t ipar ) const
{
   // return index of parameter pname
   Int_t ierr = 0;
   TString pname;
   Double_t value,verr,vlow,vhigh;  

   GetMinuit()->mnpout( ipar, pname, value, verr, vlow, vhigh, ierr );
   return pname;   
}

Int_t Gfitter::GMinuitFitter::SetParameter( Int_t ipar, const char *parname,
                                            Double_t value, Double_t verr, Double_t vlow, Double_t vhigh )
{
   // set initial values for a parameter
   if (m_fitCovMat) { delete [] m_fitCovMat; m_fitCovMat = 0; }
   Int_t ierr = 0;
   GetMinuit()->mnparm( ipar, parname, value, verr, vlow, vhigh, ierr );
  
   return ierr;   
}

Int_t Gfitter::GMinuitFitter::GetErrors( Int_t ipar, 
                                         Double_t& eplus, Double_t& eminus, Double_t &eparab, Double_t &globcc ) const
{
   // return current errors for a parameter
   Int_t ierr = 0;
   GetMinuit()->mnerrs( ipar, eplus, eminus, eparab, globcc );
   return ierr;   
}

Double_t Gfitter::GMinuitFitter::GetCovarianceMatrixElement( Int_t i, Int_t j ) 
{
   // return element i,j from the covariance matrix
   Int_t npars = GetMinuit()->GetNumPars();
   if (!m_fitCovMat) m_fitCovMat = new Double_t[npars*npars];
   GetMinuit()->mnemat( m_fitCovMat, npars );

   // sanity check
   if (i < 0 || i >= npars || j < 0 || j >= npars) {
      m_logger << kFATAL << "<GetCovarianceMatrixElement>: Illegal arguments: " << i << " " << j << GEndl;
   }
   return m_fitCovMat[j + npars*i];   
}

void Gfitter::GMinuitFitter::Reset()
{
   // reset the fitter environment
   GetMinuit()->mncler();
   
   // reset the internal Minuit random generator to its initial state (see TFitter)
   Double_t val = 3;
   Int_t inseed = 12345;
   GetMinuit()->mnrn15( val, inseed );
}
