/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GResultScan1d                                                         *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <iostream>

#include "TGraph.h"
#include "TH1F.h"
#include "TF1.h"

#include "Gfitter/GResultScan1d.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GRootFinder.h"

using std::cout;
using std::endl;

ClassImp(Gfitter::GResultScan1d)

Gfitter::GResultScan1d::GResultScan1d()
   : GResultBase(),
     m_hist ( 0 ),
     m_graph( 0 ),
     m_func ( 0 )
{
   m_thisPointer = this;
}

Gfitter::GResultScan1d::GResultScan1d( TH1F* h )
   : GResultBase(),
     m_hist ( h ),
     m_graph( new TGraph( m_hist ) )
{
   m_thisPointer = this;
   m_func = new TF1( "IGetGraphValForFunction", 
                     &IGetGraphValForFunction, 
                     m_hist->GetBinCenter(m_hist->GetMinimumBin()-1), 
                     m_hist->GetBinCenter(m_hist->GetMinimumBin()+1),0)   ;
}

Gfitter::GResultScan1d::~GResultScan1d()
{
   if (m_graph) delete m_graph;
   if (m_func)  delete m_func;
}
 
Double_t Gfitter::GResultScan1d::GetChi2Min() const
{
   // calculate the minimum of the function, use estimate boundaries of m_hist
   Double_t result = m_func->GetMinimum(m_hist->GetBinCenter(m_hist->GetMinimumBin()-1),
                                        m_hist->GetBinCenter(m_hist->GetMinimumBin()+1));
   return result;
}

Double_t Gfitter::GResultScan1d::GetValue() const 
{
   // calculate the the X value a the minimum of the function, use estimate boundaries of m_hist
   Double_t result =  m_func->GetMinimumX(m_hist->GetBinCenter(m_hist->GetMinimumBin()-1),
                                          m_hist->GetBinCenter(m_hist->GetMinimumBin()+1));
   
   return result;
}

Double_t Gfitter::GResultScan1d::GetErrGaussSym(UInt_t nsigma) const 
{
   // calculate the mean of asymetric errors
   return (this->GetErrGaussAsym(nsigma) - this->GetErrGaussAsym(-nsigma))/2.; 
}

Double_t Gfitter::GResultScan1d::GetErrGaussAsym(Int_t nsigma) const 
{
   // calculate the uncertainty of n sigma
   // n is signed! left or right of minimum.
   Double_t chi2=0.;
   Double_t low = 0.;
   Double_t high = 0.;

   // chi2 value for which the x value is searched for
   chi2 = ( TMath::Abs(nsigma)*TMath::Abs(nsigma) ) + this->GetChi2Min();

   // get initial interval estimate from m_hist
   if (nsigma<=0) {
      // left of minimum
      for (int i=m_hist->GetMinimumBin(); i>0; i--) {
         if ( m_hist->GetBinContent(i) > (chi2 ) ) {
            low = m_hist->GetBinCenter(i+1);
            high = m_hist->GetBinCenter(i);
            break;
         }
      }
   } 
   else {
      // right of minimum
      for (int i=m_hist->GetMinimumBin(); i<m_hist->GetNbinsX(); i++) {
         if ( m_hist->GetBinContent(i) > (chi2 ) ) {
            low = m_hist->GetBinCenter(i-1);
            high = m_hist->GetBinCenter(i);
            break;
         }
      }
   } 

   // calculate precise value using root finding of graph (spline) 
   return this->GetXValue( chi2, low, high ) - this->GetValue(); 
}

Double_t Gfitter::GResultScan1d::GetGlobCorr() const 
{
   // sensible ?
   return 0; 
}

// root finding
Gfitter::GResultScan1d* Gfitter::GResultScan1d::m_thisPointer = NULL;

Double_t Gfitter::GResultScan1d::IGetGraphValForRoot( Double_t x )
{
   // interface for RootFinder
   return GResultScan1d::GetThisPointer()->GetGraph()->Eval( x,0,"S" );
}

Double_t Gfitter::GResultScan1d::IGetGraphValForFunction( Double_t* x, Double_t* /* par */ )
{
   // interface for TF1
   return GResultScan1d::GetThisPointer()->GetGraph()->Eval( x[0],0,"S" );
}

Double_t Gfitter::GResultScan1d::GetXValue( Double_t chi2, Double_t low, Double_t high ) const 
{
   GRootFinder rootFinder( &IGetGraphValForRoot, low, high );
   return rootFinder.Root(chi2);   
}

