/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GScalerBase                                                           *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/

#include <iostream>

#include "TString.h"
#include "TMath.h"

#include "Gfitter/GScalerBase.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GData.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GStringList.h"

using namespace std;

Gfitter::GScalerBase::GScalerBase( const TString& expression, Double_t errp, Double_t errm, Int_t sign )
   : m_parName   ( "" ),
     m_expression( "" ),
     m_errp      ( errp ),
     m_errm      ( errm ),
     m_sign      ( sign ),
     m_par       ( 0 ),
     m_refData   ( 0 ),
     m_refName   ( "" ),
     m_first     ( kTRUE ),
     m_denomP    ( 0 ),
     m_denomM    ( 0 ),
     m_ref       ( 0 ),
     m_logger    ( GMsgLogger( "GScalerBase" ) )
{
   // scaler string must be of the form "par, par=x +- y"
   if (!expression.Contains(",")) {
      m_logger << kFATAL << "Mismatch (1) in scaler expression: \"" << expression << "\": missing \',\'" << GEndl;
   }

   // interpret the expression
   GStringList sl1( expression, "," );
   if (sl1.GetEntries() != 2) {
      m_logger << kFATAL << "Mismatch (2) in scaler expression: \"" << expression << "\": missing \',\'" << GEndl;
   }

   // first entry is formula
   m_parName = m_expression = sl1[0];

   // second entry is reference value (format, eg, "beta=0.5 +- 0.1")
   GStringList sl2(  sl1[1], "=" );
   m_refName = sl2[0]; m_refName.ReplaceAll(" ","");

   Double_t rval, rerrp, rerrm;

   GUtils::InterpretValueErrorString( sl2[1], rval, rerrp, rerrm, m_logger );
   m_refData = new GData( rval, rerrp, rerrm );
   m_logger << kINFO 
            << "Instantiate scaler: \"" << m_refName << "\" with expression \"" << m_expression << "\""
            << " and reference value: " << rval << " +" << rerrp << " -" << rerrm << GEndl;
}

Gfitter::GScalerBase::GScalerBase( const GScalerBase& other )
   : m_parName   ( other.m_parName ),
     m_expression( other.m_expression ),
     m_errp      ( other.m_errp ),
     m_errm      ( other.m_errm ),
     m_sign      ( other.m_sign ),
     m_par       ( other.m_par  ),
     m_refData   ( other.m_refData ),
     m_refName   ( other.m_refName ),
     m_first     ( other.m_first ),
     m_denomP    ( other.m_denomP ),
     m_denomM    ( other.m_denomM ),
     m_ref       ( other.m_ref )
{}

Gfitter::GScalerBase::~GScalerBase()
{}
 
// operators
Bool_t Gfitter::GScalerBase::operator == (const GScalerBase&    other) const 
{ 
   return m_parName == other.GetParName(); 
}

Bool_t Gfitter::GScalerBase::operator != (const GScalerBase&    other) const 
{ 
   return m_parName != other.GetParName(); 
}

Bool_t Gfitter::GScalerBase::operator == (const GParameter& other) const 
{ 
   return m_parName == other.GetParName(); 
}

Bool_t Gfitter::GScalerBase::operator != (const GParameter& other) const 
{ 
   return m_parName != other.GetParName(); 
}

void Gfitter::GScalerBase::SetParameter( const GParameter* par )
{ 
   // parameter must be unique !
   if (m_par && m_par != par) {
      m_logger << kFATAL << "Huge troubles in <GScalerBase::SetParameter>: tried to add different parameters: "
               << m_par->GetParName() << " != " << par->GetParName() << GEndl;
   }
   else m_par = par; 
}

Bool_t Gfitter::GScalerBase::SanityCheck( const GData* data ) 
{
   // series of sanity checks
   if (!data) {
      m_logger << kWARNING 
               << "<SanityCheck> Casted GData object has null pointer. "
               << "Note that the rescaling does not support DataCurves !" << GEndl;
      return kFALSE;
   }

   if (data->GetErrRangep() != 0 || data->GetErrRangem() != 0) {
      m_logger << kWARNING 
               << "<SanityCheck> GData object has theory errors --> not supported ! " << GEndl;
      return kFALSE;
   }

   if (data->GetErrGaussp() == 0 || data->GetErrGaussm() == 0) {
      m_logger << kWARNING 
               << "<SanityCheck> GData object cannot have zero Gaussian errors ! " << GEndl;
      return kFALSE;
   }

   if (TMath::Abs(m_sign) != 1) {
      m_logger << kWARNING 
               << "<SanityCheck> Correlation sign must be positive ! " << GEndl;
      return kFALSE;
   }

   // call only first time
   m_first = kFALSE;

   return kTRUE;
}

void Gfitter::GScalerBase::SetRefValue( Double_t v ) 
{ 
   m_refData->SetValue( v ); 
   m_first = kTRUE; // bias calculation needs to be redone
}
