/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GScalerFormula                                                        *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <iostream>

#include "TString.h"
#include "TMath.h"
#include "TFormula.h"

#include "Gfitter/GScalerFormula.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GData.h"
#include "Gfitter/GTheory.h"

using namespace std;

Gfitter::GScalerFormula::GScalerFormula( const TString& expression, Double_t errp, Double_t errm, Int_t sign )
   : GScalerBase ( expression, errp, errm, sign ),
     m_formula   ( 0 )
{
   m_logger.SetSource( "GScalerFormula" );
}

Gfitter::GScalerFormula::GScalerFormula( const GScalerFormula& other )
   : GScalerBase ( other ),
     m_formula   ( other.m_formula )
{}

Gfitter::GScalerFormula::~GScalerFormula()
{
   if (m_formula) delete m_formula;
}
 
void Gfitter::GScalerFormula::SetParameter( const GParameter* par )
{
   // init only once
   if (m_formula) return;

   // the parameter that owns the scaler
   GScalerBase::SetParameter( par );   
   m_parName = par->GetParName();
   
   // should be the same as parameter name of reference data
   if (m_parName != m_refName) {
      m_logger << kFATAL << "Mismatch in scaler names: par-name != ref-name (" 
               << m_parName << " != " << m_refName << ")" << GEndl;
   }

   // intepret expression
   m_expression.ReplaceAll( par->GetFullName(), "x" );
   m_formula = new TFormula( m_parName, m_expression );   

   if (m_formula->Compile() != 0) {
      m_logger << kFATAL << "Formula expression: \"" << m_expression 
               << "\" for scaler parameter \"" << m_parName << "\" " 
               << "could not be properly compiled." << GEndl;   
   }
}

Double_t Gfitter::GScalerFormula::GetBias() 
{
   // first call of bias function -> compute fixed offset
   if (m_first) {

      if (!SanityCheck( m_refData )) m_logger << kFATAL << "Sanity check failed" << GEndl;
      
      // ----- compute fixed offset 
      
      // reference value for constant offset
      m_ref    = m_formula->Eval(m_refData->GetValue());
      m_denomP = TMath::Abs( m_formula->Eval(m_refData->GetValue() + m_refData->GetErrGaussp()) - 
                             m_formula->Eval(m_refData->GetValue()) );
      m_denomM = TMath::Abs( m_formula->Eval(m_refData->GetValue() - m_refData->GetErrGaussm()) - 
                             m_formula->Eval(m_refData->GetValue()) );
   }
   
   // ----- compute current deviation

   Double_t diff = m_formula->Eval( m_par->GetFitValue() ) - m_ref;
   Double_t bias = (diff > 0 ? m_errp*diff/m_denomP : m_errm*diff/m_denomM);

   return bias;
}

