/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GScanAnalysis                                                          *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <iostream>
#include "TH1.h"
#include "TMath.h"

#include "Gfitter/GScanAnalysis.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GCorrGaussGen.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GFitterBase.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GResultWriterBase.h"
#include "Gfitter/GResultWriterScreen.h"
#include "Gfitter/GTimer.h"

using namespace std;

Gfitter::GScanAnalysis::GScanAnalysis()
{
   InitClassName( "GScanAnalysis" );
}

Gfitter::GScanAnalysis::~GScanAnalysis()
{}

void Gfitter::GScanAnalysis::ExecuteScan( const TString& par, Int_t nbins )
{
   GParameter* gpar = gStore()->GetParameter( par );
   if (gpar == 0) {
      m_logger << kFATAL 
               << "<Scan1D> Unknown scan parameter \"" << par << "\": " << " check datacard" << GEndl;
   }     
   gpar->SetNbins( nbins );
   ExecuteScan( gpar );
}

void Gfitter::GScanAnalysis::ExecuteScan( GParameter* gpar )
{
   // perform scan of variable to estimate such things like the influence of theoretical errors
   // the following steps are performed:
   // 1) set the scanned variable on fixed value
   // 2) perform fit with scanned variable fixed
   // 3) compute all variables
   // 4) release scanned variable, and fit again
   // 5) return to 1)

   // create histogram for scanned variable
   if (gpar == 0) m_logger << kFATAL << "<Scan1D> Zero GParameter pointer" << GEndl;

   m_logger << kINFO << "Scan analysis of parameter \"" << gpar->GetParName() << "\""
            << " (full name: \"" << gpar->GetFullName() << "\")"
            << " in interval [" << gpar->GetScanRange().GetMin() << ", " << gpar->GetScanRange().GetMax() << "]"
            << GEndl;
   m_logger << kINFO << "Number of steps in scan: " << gpar->GetNbins() << GEndl;

   // backup paramter settings
   gpar->Backup();
  
   // create histograms (also serves as scan axis)
   TString bulkName = gpar->GetFullNameWONSpace();
   TH1F* th1Chi2  = new TH1F( bulkName + (TString)"_chi2", 
                              bulkName + (TString)" (chi-squared)", 
                              gpar->GetNbins(), gpar->GetScanRange().GetMin(), gpar->GetScanRange().GetMax() );
   TH1F* th1mean  = new TH1F( bulkName + (TString)"_mean", 
                              bulkName + (TString)" (fit mean)", 
                              gpar->GetNbins(), gpar->GetScanRange().GetMin(), gpar->GetScanRange().GetMax() );
   TH1F* th1errp  = new TH1F ( bulkName + (TString)"_errp", 
                               bulkName + (TString)" (positive fit error)", 
                               gpar->GetNbins(), gpar->GetScanRange().GetMin(), gpar->GetScanRange().GetMax() );
   TH1F* th1errm  = new TH1F ( bulkName + (TString)"_errm", 
                               bulkName + (TString)" (negative fit error)", 
                               gpar->GetNbins(), gpar->GetScanRange().GetMin(), gpar->GetScanRange().GetMax() );
   TH1F* th1err   = new TH1F ( bulkName + (TString)"_err", 
                               bulkName + (TString)" (average fit error)", 
                               gpar->GetNbins(), gpar->GetScanRange().GetMin(), gpar->GetScanRange().GetMax() );

   // timing
   Int_t iloop = 0;
   GTimer timer( 2*gpar->GetNbins(), GetPrintName() );

   // two scans (foreward and backwards, to improve fit convergence)
   m_logger << kINFO << "Scanning ... please be patient" << GEndl;
         
   for (Int_t ibin=1; ibin != th1Chi2->GetNbinsX()+1; ibin++) {
 
      // clarify for later that this parameter is going to be scanned
      gpar->SetScanned( kTRUE );      
         
      // 1) initialise free parameters in fitter, and execute fit
      gStore()->GetFitter()->Initialise();         
      gpar->SetFitValue( th1Chi2->GetBinCenter( ibin ) );
      
      // step 2) perform the fit with "gpar" fixed
      Double_t chi2  = gStore()->GetFitter()->ExecuteFit();
      gStore()->GetFitter()->UpdateResults();
      
      // step 3) compute values of observables with theories from theory prodiction and  
      //         store as new central value
      GParPtrVec_t::const_iterator activePar = gStore()->GetActiveParameters().begin();
      for (; activePar != gStore()->GetActiveParameters().end(); activePar++) {
         if ((*activePar)->HasTheory()) (*activePar)->SetValue( (*activePar)->GetTheoryPrediction() );
      }
      
      // step 4) release "gpar" and refit (chi2 should be zero)
      gpar->SetScanned( kFALSE );
      gStore()->GetFitter()->Initialise();         
      chi2  = gStore()->GetFitter()->ExecuteFit();
      gStore()->GetFitter()->UpdateResults();

      GResultWriterBase* rw = new GResultWriterScreen();
      rw->AddResults( "Evaluate" );      
      delete rw;

      m_logger << kINFO 
               << Form( "\"%s\" (bin:%4i) | chi2: %7.4g | value: %8.5g (fit: %8.5g) | errors: +%8.5g -%8.5g | time left: %s",
                        gpar->GetFullName().Data(), ibin, chi2, 
                        th1Chi2->GetBinCenter( ibin ), gpar->GetFitValue(), 
                        TMath::Abs(gpar->GetResult()->GetErrGaussAsym( +1 )), 
                        TMath::Abs(gpar->GetResult()->GetErrGaussAsym( -1 )),
                        timer.GetLeftTime( ++iloop ).Data() )
               << GEndl;
    
      // fill histograms
      th1Chi2->SetBinContent( ibin, chi2 );
      th1mean->SetBinContent( ibin, gpar->GetFitValue() );
      th1errp->SetBinContent( ibin, gpar->GetResult()->GetErrGaussAsym( +1 ) );
      th1errm->SetBinContent( ibin, gpar->GetResult()->GetErrGaussAsym( -1 ) );
      th1err ->SetBinContent( ibin, 0.5*(TMath::Abs(gpar->GetResult()->GetErrGaussAsym( -1 )) + 
                                         TMath::Abs(gpar->GetResult()->GetErrGaussAsym( +1 ))) );
   }

   th1Chi2->Write();
   th1mean->Write();
   th1errp->Write();
   th1errm->Write();
   th1err ->Write();

   m_logger << kINFO << "Elapsed time: " << timer.GetElapsedTime() 
            << "                                      " << GEndl;    
}

ostream& Gfitter::operator << ( ostream& os, const Gfitter::GScanAnalysis& /* toy */ ) 
{
   os << 0;
   return os;
}
