/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GStore                                                                *
 *                                                                                *
 * Description:                                                                   *
 *      Global Dictionary                                                         *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef Gfitter_GStore
#define Gfitter_GStore

#include <vector>
#include <map>

#include "TMatrixD.h"

#include "Gfitter/GObject.h"

class TString;
class TFile;

namespace Gfitter {

   // forward declarations
   class GParameter;
   class GStore;
   class GVariable;
   class GFitterBase;
   class GAction;

   // global types and functions
   typedef std::vector<Gfitter::GParameter*> GParPtrVec_t;

   std::ostream& operator << ( std::ostream& o, const GStore& s );
   GStore* gStore();

   class GStore : public GObject {

   public:

      // singleton
      static GStore* Instance();
      static GStore* InitialInstance(); // returns object, without creating it
    
      // accessors
      const GParPtrVec_t& GetParameters() const { return m_parameters; }
      const std::map<const TString, const Gfitter::GVariable*>& GetVariables() const { return m_variables; }

      // parameter lists
      const GParPtrVec_t& GetActiveParameters();
      const GParPtrVec_t& GetActiveUncorrelatedParameters();
      const GParPtrVec_t& GetActiveCorrelatedParameters();

      UInt_t      GetNumOfParameters()                    const { return (UInt_t)m_parameters.size(); }
      UInt_t      GetNumOfActiveParameters()              const { return (UInt_t)m_activeParameters.size(); }
      UInt_t      GetNumOfActiveParametersWithoutTheory() const;

      // parameters
      GParameter* GetParameter         ( const TString& name )  const; // check among all parameters
      GParameter* GetActiveParameter   ( const TString& name )  const; // check among active parameters only
      GParameter* GetParameterFromAlias( const TString& alias ) const;
      void        AddParameter         ( GParameter* );

      // correlations
      Double_t    GetCorrelation( GParameter* gpar1, GParameter* gpar2 );
      void        AddCorrelation( const TString& par1, const TString& par2, Double_t rho );
      void        AddCorrelation( GParameter* gpar1, GParameter* gpar2, Double_t rho );
      // correlation and covariance matrices 
      TMatrixD*   GetCorrelationMatrix( const GParPtrVec_t& ); 
      TMatrixD*   GetCovarianceMatrix ( const GParPtrVec_t& ); 

      Bool_t      IsCorrelationActive() const { return m_correlationIsActive; }
      void        SetIsCorrelationsActive( Bool_t a ) { m_correlationIsActive = a; }

      // initialisation
      void        InitAllParameters();

      // actions
      void                               AddAction( const GAction* );
      const std::vector<const GAction*>& GetActions() const { return m_actions; }

      // global message level for output
      void        SetMsgLevel( GMsgLevel mlevel ) { m_msgLevel = mlevel; }
      GMsgLevel   GetMsgLevel() const { return m_msgLevel; }      

      // variables
      Bool_t           ExistVariable     ( const TString& key );
      const GVariable* GetVariable       ( const TString& key );
      const GVariable* GetVariableIfExist( const TString& key );
      void             AddVariable       ( const GVariable* );
      void             AddVariable       ( TString key, TString value ) ;
      void             SetVariable       ( TString key, TString value ) ;

      // fitter
      GFitterBase* GetFitter() const { return m_fitter; }
      void         SetFitter( GFitterBase* fitter ) { m_fitter = fitter; }

      GFitterBase* GetPreFitter() const { return m_prefitter; }
      void         SetPreFitter( GFitterBase* prefitter ) { m_prefitter = prefitter; }

      // references to target ROOT file
      TFile*       GetTargetRootFile() const         { return m_targetRootFile;  }
      void         SetTargetRootFile( TFile* tfile ) { m_targetRootFile = tfile; }      

      //global factor used in calculation of the penalty
      Double_t    GetPenaltyFactor() const        { return m_penaltyFactor; }      
      void        SetPenaltyFactor( Double_t penaltyFactor ) { m_penaltyFactor = penaltyFactor; }

      // find any parameter in string (if not found return NULL)
      // NOTE: searches for equivalence of full name
      GParameter* SearchParameterInString( const TString& ) const;

      // create backup of all parameters in GStore
      void         BackupParameters();
      void         RecoverParameters();

      // dump all parameters
      void         DumpParameters();

      // clear the full storage (delete all objects)
      void         ClearStore();
      void         ClearParameters();

   private:

      GStore();
      ~GStore();

   private:

      // pointer to singleton instance
      static GStore* m_instance;

      // parameters
      GParPtrVec_t   m_parameters;
      GParPtrVec_t   m_activeParameters;

      // correlations between parameters
      const TString GetCorrName( const GParameter*, const GParameter* ) const;
      Bool_t                            m_correlationIsActive;
      std::map<const TString, Double_t> m_correlations;
      TMatrixD*                         m_correlationMatrix;
      TMatrixD*                         m_covarianceMatrix;
      GParPtrVec_t                      m_activeCorrelatedPars;
      GParPtrVec_t                      m_activeUncorrelatedPars;

      // variables
      std::map<const TString, const GVariable*> m_variables;

      // actions
      std::vector<const GAction*> m_actions; 

      // fitter package used
      GFitterBase* m_fitter;
      GFitterBase* m_prefitter;

      // target ROOT file
      TFile*       m_targetRootFile;

      //global factor used in calculation of the penalty
      Double_t     m_penaltyFactor;

      GMsgLevel    m_msgLevel;
   };
}

#endif
