/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GTheoryFactory                                                        *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include "Riostream.h"
#include "TString.h"
#include "TClass.h"
#include "TList.h"
#include "TObjString.h"

#include "Gfitter/GTheoryFactory.h"
#include "Gfitter/GTheoryBase.h"
#include "Gfitter/GStringList.h"

std::map<const TString, Gfitter::GTheoryBase*> Gfitter::GTheoryFactory::m_allCreatedTheories;
std::vector<Gfitter::GTheoryBase*>             Gfitter::GTheoryFactory::m_allRequestedTheories;
std::vector<Gfitter::GTheoryBase*>             Gfitter::GTheoryFactory::m_allAuxiliaryTheories;
Gfitter::GMsgLogger*                           Gfitter::GTheoryFactory::m_logger = 0;
Bool_t                                         Gfitter::GTheoryFactory::m_lockRequested = kFALSE;

Gfitter::GTheoryBase* Gfitter::GTheoryFactory::GetTheory( const TString& theoryName ) 
{
   GTheoryBase* theory = FindTheory( theoryName );
   
   if (theory!=0) return theory;
   
   // create output logger
   if (m_logger == 0) CreateMsgLogger();
   
   TString theoryBaseName( theoryName );
   TString theoryInstanceName("");
   
   // if theoryName contains ".", then an instance of a class is required
   Bool_t instanceSpecified = theoryName.Contains("/");
   
   if (instanceSpecified) {
      GStringList thlist( theoryName, "/" );
      if (thlist.GetEntries() != 2)
         *m_logger << kFATAL << "<GetTheory> Found '.' in theoryName: \""
                   << theoryName << "\" but more than one of them" << GEndl;
      theoryBaseName     =  thlist[0];
      theoryInstanceName =  thlist[1];
      *m_logger << kINFO << "Theory: \"" << theoryBaseName  << "\" has specified instance: \"" 
                << theoryInstanceName << "\"" << GEndl;
   }
   
   Bool_t isAuxiliaryTheory = kFALSE;
   TClass* tmp = gROOT->GetClass( theoryBaseName );
   if (tmp && tmp->InheritsFrom( "Gfitter::GTheoryBase" )) {	
      theory = (GTheoryBase*)tmp->New();
      if (!theory) *m_logger << kFATAL 
                             << "<GetTheory> Theory: \"" << theoryBaseName << "\" has been found "
                             << "but could not be instantiated. It might be a virtual class: did you "
                             << "implement all virtual functions defined in the \"GTheory\" base class ?"
                             << GEndl;
      if (instanceSpecified) {
         theory->SetInstanceName( theoryInstanceName );
      }
      
      isAuxiliaryTheory = theory->InheritsFrom( "Gfitter::GAuxTheory" );
      if (isAuxiliaryTheory) {
         if (!m_lockRequested) {
            // we are still parsing the datacard, no auxiliary theories should be in there
            *m_logger << kFATAL << "<GetTheory> The auxiliary theory \"" << theoryName
                      << "\" is specified in the datacard" << GEndl;
         }
         InitTheory(theory);
         m_allAuxiliaryTheories.push_back( theory );
      } 
      else {
         if (m_lockRequested) {
            // we are done parsing the datacard, only auxiliary theories can now be requested
            *m_logger << kFATAL << "<GetTheory> The proper theory \"" << theoryName 
                      << "\" is requested but it is not in the datacard" << GEndl;
         }
         AddTheoryToListOfRequested( theory );
      }
   }  
   
   if (theory == 0) {
      *m_logger << kINFO << "Parameter \"" << theoryName << "\" is not associated with a theory" << GEndl;
   }
   else {
      if (isAuxiliaryTheory) {
         *m_logger << kINFO << "\"" << theoryName << "\" is an auxiliary theory" << GEndl;
      }
      else {         
         *m_logger << kINFO << "Parameter \"" << theoryName << "\" is associated with a theory" << GEndl;
      }
   }
   
   if (theory!=0) m_allCreatedTheories[theoryName] = theory;
   
   return theory;
}

Gfitter::GTheoryBase* Gfitter::GTheoryFactory::FindTheory( const TString & thName ) 
{
   std::map<const TString, GTheoryBase*>::iterator thIt = m_allCreatedTheories.find(thName);
   if (thIt == m_allCreatedTheories.end()) return 0;
   else                                    return thIt->second;
}

void Gfitter::GTheoryFactory::AddTheoryToListOfRequested( GTheoryBase* theory ) 
{ 
   // sanity check
   if (theory == 0) {
      if (m_logger == 0) CreateMsgLogger();
      *m_logger << kFATAL << "<AddTheoryToListOfRequested> Cannot insert "
                << "zero theory pointer" << GEndl;
   } 
   else {
      if (m_logger == 0) CreateMsgLogger();
      *m_logger << kINFO << "Add theory " << theory->GetTheoryName() 
                << " to list of requested theories" << GEndl;
   }
   m_allRequestedTheories.push_back( theory ); 
}

Bool_t Gfitter::GTheoryFactory::InitTheories() 
{
   if (m_logger == 0) CreateMsgLogger();
   *m_logger << kINFO << "Initialization of " << RequestedTheories().size() << " requested theories" << GEndl;
   
   // from  this point on the creation of proper theories is forbidden,
   // only auxiliary theories can be requested from the factory
   m_lockRequested = kTRUE;
   
   Bool_t success = kTRUE;

   // initialise the parameters for all theories
   std::vector<GTheoryBase*>::const_iterator theoIt = RequestedTheories().begin();
   for (; theoIt != RequestedTheories().end(); theoIt++) {
      success *= InitTheory(*theoIt);
   }

   return success;
}

Bool_t Gfitter::GTheoryFactory::InitTheory(GTheoryBase* theory) 
{
   if (m_logger == 0) CreateMsgLogger();
   Bool_t isAuxTheory = theory->InheritsFrom( "Gfitter::GAuxTheory" );

   *m_logger << kINFO << "Initialization of " << (isAuxTheory?"auxiliary":"requested") << " theory " << theory->GetTheoryName() << GEndl;
   
   *m_logger << kINFO << "starting of InitParameters() for " << theory->GetTheoryName() << GEndl;
   Bool_t success = theory->InitParameters();
   *m_logger << kINFO << "starting of InitTheories() for " << theory->GetTheoryName() << GEndl;
   success &= theory->InitTheories();
   
   // some initialization done after construction 
   // (for instance instanceName-dependent actions)
   theory->Initialise(); 
   
   if (!success) {
      *m_logger << kERROR << "<InitTheories> Initialization of theory: \"" 
                << theory->GetTheoryName() << "\" failed" << GEndl;
   }
   *m_logger << kINFO << "leaving initialisation of " << theory->GetTheoryName() << GEndl;
   return success;
}

void Gfitter::GTheoryFactory::CreateMsgLogger()
{
   m_logger = new GMsgLogger( "GTheoryFactory" );
}

void Gfitter::GTheoryFactory::Reset()
{
   // delete all the module pntrs in the map
   while ( ! m_allCreatedTheories.empty() ) {
     std::map<const TString, Gfitter::GTheoryBase*>::iterator iter = m_allCreatedTheories.begin();
     Gfitter::GTheoryBase *mod= (*iter).second;
     m_allCreatedTheories.erase(iter);
     delete mod;
   }
   m_allRequestedTheories.clear();
   m_allAuxiliaryTheories.clear();
   m_lockRequested = kFALSE;
}

