/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Namespace: GUtils                                                              *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/

#include <iostream>
#include <iomanip>
#include <cmath>

#include "TObjString.h"
#include "TString.h"
#include "TMath.h"
#include "TList.h"
#include "TH1.h"
#include "TH2.h"
#include "TClass.h"

#include "Gfitter/GUtils.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GStringList.h"
#include "Gfitter/GVariable.h"

using namespace std;

TString Gfitter::GUtils::RemoveNamespace( const TString& inString )
{
   TString outString = inString;
   if (outString.Contains("::")) {
      Ssiz_t k = outString.First("::");
      outString.Remove(0,k+2);
   }
   return outString;
}

Float_t Gfitter::GUtils::CreateProbTHist( TH1& chi2In, TH1& dchi2Out, TH1& dlikeOut, TH1& clOut )
{
   // sanity check: number of bins and x range in both histograms must be equal
   if (chi2In.GetNbinsX() != clOut.GetNbinsX() || 
       chi2In.GetNbinsX() != dchi2Out.GetNbinsX() ||
       chi2In.GetNbinsX() != dlikeOut.GetNbinsX()) {
      cout << "Fatal error in GUtils::ComputeProbFromChi2: unequal bin numbers: " 
           << chi2In.GetNbinsX() << " != " << clOut.GetNbinsX() << " OR "
           << chi2In.GetNbinsX() << " != " << dchi2Out.GetNbinsX() << " OR "
           << chi2In.GetNbinsX() << " != " << dlikeOut.GetNbinsX() << " OR "
           << " ==> abort" << endl;
   }
   if (chi2In.GetXaxis()->GetXmin() != clOut   .GetXaxis()->GetXmin() || 
       chi2In.GetXaxis()->GetXmax() != clOut   .GetXaxis()->GetXmax() ||
       chi2In.GetXaxis()->GetXmin() != dchi2Out.GetXaxis()->GetXmin() || 
       chi2In.GetXaxis()->GetXmax() != dchi2Out.GetXaxis()->GetXmax() ||
       chi2In.GetXaxis()->GetXmin() != dlikeOut.GetXaxis()->GetXmin() ||
       chi2In.GetXaxis()->GetXmax() != dlikeOut.GetXaxis()->GetXmax()) {
      cout << "Fatal error in GUtils::ComputeProbFromChi2: unequal edges: " 
           << chi2In.GetXaxis()->GetXmin() << " != " << clOut   .GetXaxis()->GetXmin() << " OR "
           << chi2In.GetXaxis()->GetXmax() << " != " << clOut   .GetXaxis()->GetXmax() << " OR "
           << chi2In.GetXaxis()->GetXmin() << " != " << dchi2Out.GetXaxis()->GetXmin() << " OR "
           << chi2In.GetXaxis()->GetXmax() << " != " << dchi2Out.GetXaxis()->GetXmax() << " OR "            
           << chi2In.GetXaxis()->GetXmin() << " != " << dlikeOut.GetXaxis()->GetXmin() << " OR "
           << chi2In.GetXaxis()->GetXmax() << " != " << dlikeOut.GetXaxis()->GetXmax()             
           << " ==> abort" << endl;
   }

   // compute the delta_chi2 = chi2 - chi2min and then Prob in each bin
   Float_t chi2min = chi2In.GetMinimum();
   for (Int_t ibin=1; ibin<=chi2In.GetNbinsX(); ibin++) {
      Float_t dchi2 = chi2In.GetBinContent( ibin ) - chi2min;
      Float_t dlike = TMath::Exp( -0.5*dchi2 );
      dchi2Out.SetBinContent( ibin, dchi2 );
      dlikeOut.SetBinContent( ibin, dlike );
      clOut   .SetBinContent( ibin, TMath::Prob( dchi2, 1 ));
   }      

   return chi2min;
}

Float_t Gfitter::GUtils::CreateProbTHist( TH2& chi2In, TH2& dchi2Out, TH2& dlikeOut, TH2& clOut )
{
   // sanity check: number of bins and x range in both histograms must be equal
   if (chi2In.GetNbinsX() != clOut.GetNbinsX() || chi2In.GetNbinsX() != dchi2Out.GetNbinsX()) {
      cout << "Fatal error in GUtils::ComputeProbFromChi2: unequal X bin numbers: " 
           << chi2In.GetNbinsX() << " != " << clOut.GetNbinsX() << " OR "
           << chi2In.GetNbinsX() << " != " << dchi2Out.GetNbinsX() << " OR "
           << " ==> abort" << endl;
   }
   if (chi2In.GetNbinsY() != clOut.GetNbinsY() || chi2In.GetNbinsY() != dchi2Out.GetNbinsY()) {
      cout << "Fatal error in GUtils::ComputeProbFromChi2: unequal Y bin numbers: " 
           << chi2In.GetNbinsY() << " != " << clOut.GetNbinsY() << " OR "
           << chi2In.GetNbinsY() << " != " << dchi2Out.GetNbinsY() << " OR "
           << " ==> abort" << endl;
   }
   if (chi2In.GetXaxis()->GetXmin() != clOut   .GetXaxis()->GetXmin() || 
       chi2In.GetXaxis()->GetXmax() != clOut   .GetXaxis()->GetXmax() ||
       chi2In.GetXaxis()->GetXmin() != dchi2Out.GetXaxis()->GetXmin() || 
       chi2In.GetXaxis()->GetXmax() != dchi2Out.GetXaxis()->GetXmax()) {
      cout << "Fatal error in GUtils::ComputeProbFromChi2: unequal edges: " 
           << chi2In.GetXaxis()->GetXmin() << " != " << clOut   .GetXaxis()->GetXmin() << " OR "
           << chi2In.GetXaxis()->GetXmax() << " != " << clOut   .GetXaxis()->GetXmax() << " OR "
           << chi2In.GetXaxis()->GetXmin() << " != " << dchi2Out.GetXaxis()->GetXmin() << " OR "
           << chi2In.GetXaxis()->GetXmax() << " != " << dchi2Out.GetXaxis()->GetXmax()             
           << " ==> abort" << endl;
   }
   if (chi2In.GetYaxis()->GetXmin() != clOut   .GetYaxis()->GetXmin() || 
       chi2In.GetYaxis()->GetXmax() != clOut   .GetYaxis()->GetXmax() ||
       chi2In.GetYaxis()->GetXmin() != dchi2Out.GetYaxis()->GetXmin() || 
       chi2In.GetYaxis()->GetXmax() != dchi2Out.GetYaxis()->GetXmax()) {
      cout << "Fatal error in GUtils::ComputeProbFromChi2: unequal edges: " 
           << chi2In.GetYaxis()->GetXmin() << " != " << clOut   .GetYaxis()->GetXmin() << " OR "
           << chi2In.GetYaxis()->GetXmax() << " != " << clOut   .GetYaxis()->GetXmax() << " OR "
           << chi2In.GetYaxis()->GetXmin() << " != " << dchi2Out.GetYaxis()->GetXmin() << " OR "
           << chi2In.GetYaxis()->GetXmax() << " != " << dchi2Out.GetYaxis()->GetXmax()             
           << " ==> abort" << endl;
   }

   // degrees of freedom
   Int_t ndof = (gStore()->ExistVariable( "Fitter::NdofScan2d" ) ? 
                    gStore()->GetVariable( "Fitter::NdofScan2d" )->GetIntValue() : 2);
   if( ndof > 2 || ndof < 1){
      cout << "WARNING: Number of degrees of freedom = " << ndof << " ?"<< endl;
      cout << endl;
   }

   // compute the delta_chi2 = chi2 - chi2min and then Prob in each bin
   Float_t chi2min = chi2In.GetMinimum();
   for (Int_t ibin1=1; ibin1<=chi2In.GetNbinsX(); ibin1++) {
      for (Int_t ibin2=1; ibin2<=chi2In.GetNbinsY(); ibin2++) {
         Float_t chi2 = chi2In.GetBinContent( ibin1, ibin2 );
         if( isinf( chi2 ) ) chi2 = 100*chi2min;
         if( isnan( chi2 ) ) chi2 = 100*chi2min;
         Float_t dchi2 = chi2 - chi2min;
         Float_t dlike = TMath::Exp( -0.5*dchi2 );
         dchi2Out.SetBinContent( ibin1, ibin2, dchi2 );
         dlikeOut.SetBinContent( ibin1, ibin2, dlike );
         clOut .SetBinContent( ibin1, ibin2, TMath::Prob( dchi2, ndof ));
      }
   }      

   return chi2min;
}

void Gfitter::GUtils::PrintParVector( GMsgLogger& logger, const GParPtrVec_t& v, TString vecName ) 
{
   logger << kINFO << "Parameter vector \"" << vecName << "\":" << GEndl;
   Int_t ipar = 0;
   for (GParPtrVec_t::const_iterator par = v.begin(); par != v.end(); par++, ipar++) 
      logger << kINFO << "... parameter(" << ipar << "): " << (*par)->GetFullName() << GEndl;
}

void Gfitter::GUtils::FormattedOutput( const std::vector<Double_t>& values, const std::vector<TString>& V, 
                                       const TString titleVars, const TString titleValues, GMsgLogger& logger,
                                       TString format )
{
   // formatted output of simple table

   // sanity check
   UInt_t nvar = V.size();
   if ((UInt_t)values.size() != nvar) {
      logger << kFATAL << "<FormattedOutput> fatal error with dimensions: " 
             << values.size() << " OR " << " != " << nvar << GEndl;
   }

   // find maximum length in V (and column title)
   UInt_t maxL = 7;
   std::vector<UInt_t> vLengths;
   for (UInt_t ivar=0; ivar<nvar; ivar++) maxL = TMath::Max( (UInt_t)V[ivar].Length(), maxL );
   maxL = TMath::Max( (UInt_t)titleVars.Length(), maxL );

   // column length
   UInt_t maxV = 7;
   maxV = TMath::Max( (UInt_t)titleValues.Length() + 1, maxL );

   // full column length
   UInt_t clen = maxL + maxV + 3;

   // bar line
   for (UInt_t i=0; i<clen; i++) logger << "-";
   logger << GEndl;

   // title bar   
   logger << setw(maxL) << titleVars << ":";
   logger << setw(maxV+1) << titleValues << ":";
   logger << GEndl;
   for (UInt_t i=0; i<clen; i++) logger << "-";
   logger << GEndl;

   // the numbers
   for (UInt_t irow=0; irow<nvar; irow++) {
      logger << setw(maxL) << V[irow] << ":";
      logger << setw(maxV+1) << Form( format.Data(), values[irow] );
      logger << GEndl;
   }

   // bar line
   for (UInt_t i=0; i<clen; i++) logger << "-";
   logger << GEndl;
}

void Gfitter::GUtils::FormattedOutput( const TMatrixD& M, const GParPtrVec_t& gpars, 
                                       GMsgLogger& logger, const TString& format, Bool_t useFullName )
{
   std::vector<TString> pars;
   GParPtrVec_t::const_iterator it = gpars.begin();
   for (; it != gpars.end(); it++) pars.push_back( useFullName ? (*it)->GetFullName() : (*it)->GetParName() );
   FormattedOutput( M, pars, logger, format );                                                   
}

void Gfitter::GUtils::FormattedOutput( const TMatrixD& M, const std::vector<TString>& V, 
                                       GMsgLogger& logger, const TString& format )
{
   // formatted output of matrix (with labels)

   // sanity check: matrix must be quadratic
   UInt_t nvar = V.size();
   if ((UInt_t)M.GetNcols() != nvar || (UInt_t)M.GetNrows() != nvar) {
      logger << kFATAL << "<FormattedOutput> Error with dimensions: " 
             << M.GetNcols() << " OR " << M.GetNrows() << " != " << nvar << " ==> abort" << GEndl;
   }

   // get length of each variable, and maximum length  
   TString testnum = Form( format, M(0,0) );
   UInt_t minL = TMath::Max(6,(Int_t)testnum.Sizeof());
   UInt_t maxL = minL;
   std::vector<UInt_t> vLengths;
   for (UInt_t ivar=0; ivar<nvar; ivar++) {
      vLengths.push_back(TMath::Max( (UInt_t)V[ivar].Length(), minL ));
      maxL = TMath::Max( vLengths.back(), maxL );
   }
   
   // count column length
   UInt_t clen = maxL+2;
   for (UInt_t icol=0; icol<nvar; icol++) clen += vLengths[icol]+1;

   // bar line   
   for (UInt_t i=0; i<clen; i++) logger << "-";
   logger << GEndl;

   // title bar   
   logger << setw(maxL+2) << " ";
   for (UInt_t icol=0; icol<nvar; icol++) logger << setw(vLengths[icol]+1) << V[icol];
   logger << GEndl;

   // bar line
   logger << setw(maxL+1) << " ";
   logger << "+";
   for (UInt_t i=0; i<clen-(maxL+2); i++) logger << "-";
   logger << GEndl;

   // the numbers
   for (UInt_t irow=0; irow<nvar; irow++) {
      logger << setw(maxL) << V[irow] << " |";
      for (UInt_t icol=0; icol<nvar; icol++) {
         logger << setw(vLengths[icol]+1) << Form( format, M(irow,icol) );
      }      
      logger << GEndl;
   }

   // bar line
   for (UInt_t i=0; i<clen; i++) logger << "-";
   logger << GEndl;
}

void Gfitter::GUtils::CutStringInParts( const TString& in, Int_t at, TString& v1, TString& v2 )
{
   v1 = in(0,at);
   v2 = in(at,in.Sizeof()-at);   
}

void Gfitter::GUtils::InterpretValueErrorString( const TString& instr, Double_t& val, 
                                                 Double_t& errp, Double_t& errm, GMsgLogger& logger )
{
   // expected format: "x +- a" or "x +a -b" or "x -a +b"
   // separate into value and error part

   // sanity check: routine does not work with exp in string
   if (instr.Contains("e") || instr.Contains("E")) {
      logger << kFATAL << "Troubles: cannot interpret string with exp \"e\" function."
             << " This is a missing feature which must be corrected --> please contact the authors !" 
             << GEndl;
   }

   TString sep;
   if (instr.Index('+') < instr.Index('-')) sep = "+"; // format: "x +- a" or "x +a -b"
   else                                     sep = "-"; // format: "x -+ a" or "x -a +b"
   
   GStringList vl( instr, sep );
   if (vl.GetEntries() != 2) logger << kFATAL << "Huge troubles (0): \"" << instr << "\"" << GEndl;
   val = vl[0].Atof();
   
   InterpretErrorString( sep + vl[1], errp, errm, logger );
}

void Gfitter::GUtils::InterpretValueErrorsString ( const TString& instr, Double_t& val, 
                                                   vector<Double_t>& estatp, vector<Double_t>& estatm, 
                                                   vector<Double_t>& esystp, vector<Double_t>& esystm, 
                                                   vector<Double_t>& etheop, vector<Double_t>& etheom, 
                                                   GMsgLogger& logger )
{
   // format
   // D1: x +- dx_stat +- dx1_syst +- dx2_syst ... +- dt1_theo +- ... +- ds1 (scaler1) +- ds2 (scaler2) +- ...
   // including possibly asymmetric errors

   // sanity check first
   if (!instr.Contains("_stat") && !instr.Contains("_syst") && !instr.Contains("_theo")) {
      logger << kFATAL << "Cannot interpret value-error string \"" << instr << "\"" << GEndl;
   }

   // sanity check that no other alphanumerics
   TString teststr = instr; teststr.ToLower();
   TString rpl[] = { "stat", "syst", "theo", "e", "+", "-", "_", "." };
   for (Int_t i=0; i<8; i++) {
      teststr.ReplaceAll( rpl[i], " " );      
   }
   if (!teststr.IsDigit()) {
      logger << kFATAL << "Cannot interpret value-error string \"" << instr << "\"" 
             << " (after replacement of all allowed characters, found string: \"" << teststr << "\"" << GEndl;
   }

   // init vectors
   estatp.clear(); estatm.clear();
   esystp.clear(); esystm.clear();
   etheop.clear(); etheom.clear();  

   // remove value from string
   TString label;
   label.Append(instr(0));   
   Int_t pos = -1;
   for (Int_t i=1; i<instr.Length(); i++) {
      label.Append(instr(i));
      if (instr(i-1) != (TString)"e" && instr(i-1) != (TString)"E") {
         if (instr(i) == TString("+") || instr(i) == TString("-")) {
            label.Chop();
            val = label.Atof();
            pos = i;
            break;
         }
      }
   }

   TString str = instr;
   str.Remove(0,pos);

   // now "str": only contains errors
   GStringList list( str, "_" );
   for (Int_t iarg=0; iarg<list.GetEntries()-1; iarg++) {
      TString a1 = list.GetArg(iarg);
      TString a2 = list.GetArg(iarg+1);
      if (a1.Contains("stat")) a1.ReplaceAll("stat","");
      if (a1.Contains("syst")) a1.ReplaceAll("syst","");
      if (a1.Contains("theo")) a1.ReplaceAll("theo","");
      Double_t ep, em;
      InterpretErrorString( a1, ep, em, logger );
      if      (a2.Contains("stat")) { estatp.push_back(ep); estatm.push_back(em); }
      else if (a2.Contains("syst")) { esystp.push_back(ep); esystm.push_back(em); }
      else if (a2.Contains("theo")) { etheop.push_back(ep); etheom.push_back(em); }
   }
}

void Gfitter::GUtils::InterpretErrorString( const TString& instr_, Double_t& errp, Double_t& errm,
                                            GMsgLogger& logger )
{
   // expected format: "+- a" or "+a -b" or "-a +b"
   TString instr = instr_;

   // protect against exp "e" expressions
   instr.ToLower();
   instr.ReplaceAll("e+","e$p");
   instr.ReplaceAll("e-","e$m");

   if (instr.Contains("+-")) {
      // symmetric errors
      instr.ReplaceAll("+-","");
      instr.ReplaceAll("e$p","e+"); instr.ReplaceAll("e$m","e-");
      errp = errm = instr.Atof();
   }
   else if (instr.Contains("+") && instr.Contains("-")) {
      // asymmetric errors
      if (instr.Index('+') < instr.Index('-')) {
         // format: "+a -b"
         GStringList vl( instr, "-" );
         if (vl.GetEntries() != 2) logger << kFATAL << "Huge troubles (1): \"" << instr << "\"" << GEndl;

         TString sp = vl[0];
         sp.ReplaceAll("+","");
         sp.ReplaceAll("e$p","e+"); sp.ReplaceAll("e$m","e-");
         errp = sp.Atof();

         TString sm = vl[1];
         sm.ReplaceAll("e$p","e+"); sm.ReplaceAll("e$m","e-");
         errm = sm.Atof();
      }
      else {
         // format: "-a +b"
         GStringList vl( instr, "+" );
         if (vl.GetEntries() != 2) logger << kFATAL << "Huge troubles (2): \"" << instr << "\"" << GEndl;

         TString sm = vl[0];
         sm.ReplaceAll("-","");
         sm.ReplaceAll("e$p","e+"); sm.ReplaceAll("e$m","e-");
         errm = sm.Atof();

         TString sp = vl[1];
         sp.ReplaceAll("e$p","e+"); sp.ReplaceAll("e$m","e-");
         errp = sp.Atof();
      }
   }
   else logger << kFATAL << "Big troubles in ErrScale string: \"" << instr_ << GEndl;
}

TString Gfitter::GUtils::ExtractValueFromParSetting( const TString& str, GMsgLogger& logger )
{
   GStringList gl( str, "=" );
   if (gl.GetEntries() != 2) logger << kFATAL 
                                    << "<GUtils::ExtractValueFromParSetting>: Wrong par setting format: "
                                    << str << ", expected: \"par = value\"" 
                                    << GEndl;

   return gl[1];
}

// replace regular expressions
TString Gfitter::GUtils::ReplaceRegularExpressions( const TString& s, const TString& r )  
{
   // helper function to remove all occurences "!%^&()'<>?= " from a string
   // and replace all ::,*,/,+,- with _M_,_T_,_D_,_P_,_M_ respectively

   static const TString regexp( "!%^&()'<>?=*/+-[] " );
   
   TString snew = s;
   for (Int_t i = 0; i < regexp.Length(); i++) snew.ReplaceAll( regexp[i], r );

   return snew;
}


double 
Gfitter::ground(double x, unsigned int digits) {
  if (digits > 0) {
    return ground(x*10.0, digits-1)/10.0;
  }
  else {
    return round(x);
  }
}


