/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GWorkspace                                                            *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/

#include <cmath>
#include <iostream>

#include "TROOT.h"
#include "TFile.h"

//#include "RooWorkspace.h"
//#include "RooStats/ModelConfig.h"

#include "Gfitter/GWorkspace.h"
#include "Gfitter/GMsgLogger.h"

#include "RooAbsPdf.h"
#include "RooAbsData.h"
#include "RooArgSet.h"
#include "RooRealVar.h"

using namespace std;

ClassImp(Gfitter::GWorkspace)

//_____________________________________________________________________________
Gfitter::GWorkspace::GWorkspace()
   : GDataBase()
{
   InitClassName( "GWorkspace" );   
}

//_____________________________________________________________________________
Gfitter::GWorkspace::GWorkspace( const TString& fileName, const TString& workspaceName, const TString& modelconfigstr )
  : GDataBase()
  , m_modelconfigstr(modelconfigstr)
{
  InitClassName( "GWorkspace" );   

  if (fileName.IsNull()) {
    m_logger << kFATAL << "Invalid file name." << GEndl;
  }
  if (workspaceName.IsNull()) {
    m_logger << kFATAL << "Invalid workspace name." << GEndl;
  }
  if (modelconfigstr.IsNull()) {
    m_logger << kFATAL << "Invalid workspace name." << GEndl;
  }
  
  m_logger << kINFO << "Initializing a GWorkspace" << GEndl;
  m_logger << kINFO 
	   << "Root File: \"" << fileName 
	   << "\" Workspace: \"" << workspaceName 
	   << "\" Model config: \"" << modelconfigstr 
	   << "\"" << GEndl;
  
  m_workspace = this->GetWorkspaceFromFile(fileName,workspaceName);
  (void) this->SetModelConfig();
  (void) this->ConstructPLL();
}

//_____________________________________________________________________________
Gfitter::GWorkspace::GWorkspace( const GWorkspace& other )
  : GDataBase( other )
  , m_modelconfigstr( other.m_modelconfigstr )
{   
  InitClassName( "GWorkspace" );   
  
  m_workspace = new RooWorkspace(*other.m_workspace);
  this->SetModelConfig();

  // HACK : copied workspace has dataset without name
  TString datasetname = other.m_modelconfig->GetProtoData()->GetName();
  m_workspace->data("")->SetName(datasetname.Data());

  (void) this->ConstructPLL();
}

//_____________________________________________________________________________
Gfitter::GWorkspace::~GWorkspace()
{
  if (m_pll!=0) { delete m_pll; m_pll=0; }
  if (m_nll!=0) { delete m_nll; m_nll=0; }
  if (m_modelconfig!=0) { delete m_modelconfig; m_modelconfig=0; }  
  if (m_workspace!=0) { delete m_workspace; m_workspace=0; }
}
 

//_____________________________________________________________________________
Double_t 
Gfitter::GWorkspace::GetChiSquared() const
{
  Double_t retVal = 0.0;

  if (m_pll!=0) retVal = 2.*m_pll->getVal(); 

  return retVal;
}

//________________________________________________________________________________________________
RooWorkspace*
Gfitter::GWorkspace::GetWorkspaceFromFile( const TString& infile, const TString& wsname )
{
  TFile* file = TFile::Open(infile.Data(), "READ");
  if (file->IsZombie()) {
    m_logger << kFATAL << "Cannot open file: " << infile << GEndl;
    return NULL;
  }
  file->cd();

  TObject* obj = file->Get( wsname.Data() ) ;
  if (obj==0) {
    m_logger << kFATAL << "Cannot open workspace <" << wsname << "> in file <" << infile << ">" << GEndl;
    file->Close();
    return NULL;
  }

  if (obj->ClassName()!=TString("RooWorkspace")) {
    m_logger << kFATAL << "Cannot open workspace <" << wsname << "> in file <" << infile << ">" << GEndl;
    file->Close();
    return NULL;
  }

  RooWorkspace* w = (RooWorkspace*)( obj );
  if ( w==0 ) {
    m_logger << kFATAL << "ERROR : Cannot open workspace <" << wsname << "> in file <" << infile << ">" << GEndl;
    file->Close();
    return NULL;
  }

  file->Close();
  return w;
}

//________________________________________________________________________________________________
Bool_t
Gfitter::GWorkspace::SetModelConfig()
{
  TObject* obj = m_workspace->obj( m_modelconfigstr ) ;
  if (obj==0) {
    m_logger << kFATAL << "No object <" << m_modelconfigstr << "> in workspace." << GEndl;
    return false;
  }
  if (obj->ClassName()!=TString("RooStats::ModelConfig")) {
    m_logger << kFATAL << "Object <" << m_modelconfigstr << ">not of type ModelConfig" << GEndl;
    return false;
  }

  m_modelconfig = (RooStats::ModelConfig*)( obj );
  if ( m_modelconfig==0 ) {
    m_logger << kFATAL << "Object <" << m_modelconfigstr << ">not of type ModelConfig" << GEndl;
    return false;
  }

  return true;
}

//________________________________________________________________________________________________
Bool_t 
Gfitter::GWorkspace::ConstructPLL()
{
  RooAbsPdf*  pdf  = m_modelconfig->GetPdf();
  RooAbsData* data = m_modelconfig->GetProtoData();
  const RooArgSet*  poi  = m_modelconfig->GetParametersOfInterest();

  m_nll = pdf->createNLL(*data);
  m_pll = m_nll->createProfile(*poi) ;
  
  return true;
}

//________________________________________________________________________________________________
Bool_t
Gfitter::GWorkspace::SetParInWS(const TString& parname, const Double_t& value)
{
  RooRealVar* par = m_workspace->var(parname.Data());

  if (par==0) {
    m_logger << kFATAL << "RooRealVar <" << parname << "> not found in workspace." << GEndl;
    return false;
  } else {
    par->setVal(value);
  }

  return true;
}

///////////////////////////////////////////////////////////////////////////////

//_____________________________________________________________________________
Double_t 
Gfitter::GWorkspace::GetValue() const
{
   // move minimum of histogram here
   m_logger << kFATAL << "getting of values for GWorkspace still to be implemented" << GEndl;
   return 0.0;
}

//_____________________________________________________________________________
Bool_t 
Gfitter::GWorkspace::SetValue( Double_t ) 
{
   // move minimum of histogram here
   m_logger << kFATAL << "setting of values for GWorkspace still to be implemented" << GEndl;
   return 0;
}

//_____________________________________________________________________________
Double_t 
Gfitter::GWorkspace::GetErrGaussp() const {
   // m_logger << kFATAL << "GetErrGaussp not yet implemented for GWorkspace" << GEndl;
   // MG: Set Error to 0
   return 0.;
}

//_____________________________________________________________________________
Double_t 
Gfitter::GWorkspace::GetErrGaussm() const {
   //m_logger << kFATAL << "GetErrGaussm not yet implemented for GWorkspace" << GEndl;
   // MG: Set Error to 0
   return 0.;
}

//_____________________________________________________________________________
Double_t 
Gfitter::GWorkspace::GetErrTotp() const {
   m_logger << kFATAL << "GetErrTotp not yet implemented for GWorkspace" << GEndl;
   return 0.;
}

//_____________________________________________________________________________
Double_t 
Gfitter::GWorkspace::GetErrTotm() const {
   m_logger << kFATAL << "GetErrTotm not yet implemented for GWorkspace" << GEndl;
   return 0.;
}

//_____________________________________________________________________________
Double_t 
Gfitter::GWorkspace::GetErrTotSym() const {
   m_logger << kFATAL << "GetErrTotSym not yet implemented for GWorkspace" << GEndl;
   return 0.;
}

//_____________________________________________________________________________
Double_t 
Gfitter::GWorkspace::GetRndValue( Double_t /*mean*/ )
{
   m_logger << kFATAL << "GetGaussRnd is not defined for GWorkspace" << GEndl;
   return 0.;
}

//_____________________________________________________________________________
Double_t 
Gfitter::GWorkspace::GetGaussRnd( Double_t ) const
{
   m_logger << kFATAL << "GetGaussRnd is not defined for GWorkspace" << GEndl;
   return 0.;
}

