/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : QPoleMass                                                             *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory for inverting Ms-bar masses to pole quark masses         *
 *                                                                                *
 * Source:                                                                        *
 *     (i) K. G. Chetyrkin and M. Steinhauser                                     *
 *         Nucl. Phys.  B 573 , 617 (2000), [arXiv:hep-ph/9911434]                *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *     
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"

#include "GEW/QPoleMass.h"
#include "GEW/RunningAlphaQCD.h"
#include "GEW/AlphaQCDAtQ.h"

using namespace Gfitter;

ClassImp(GEW::QPoleMass)

GEW::QPoleMass::QPoleMass()
   : GAuxTheory()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );
   
   BookParameter( "mu_MSb", &p_mu );   
   BookParameter( "md_MSb", &p_md );   
   BookParameter( "ms_MSb", &p_ms );   
   BookParameter( "mc_MSb", &p_mc );   
   BookParameter( "mb_MSb", &p_mb ); 

   BookTheory( "GEW::RunningAlphaQCD", & t_AlphasRun );
}

// MS_bar to pole mass
Double_t GEW::QPoleMass::MSMassToPoleMass( Double_t m, Int_t nf )
{
   Double_t As        = 0;
   Double_t nl        = nf - 1;

   switch( nf ){
   case 1:
      As  = GetAlphasRun().EvolveAlphas(1)/TMath::Pi();
      break;
   case 2:
      As  = GetAlphasRun().EvolveAlphas(1)/TMath::Pi();
      break;
   case 3:
      As  = GetAlphasRun().EvolveAlphas(1)/TMath::Pi();
      break;
   case 4:
      if (TMath::IsNaN(p_mc)) m_logger << kFATAL << "<QPoleMass::MSMassToPoleMass> p_mc is NaN !" << GEndl;
      As  = GetAlphasRun().EvolveAlphas(p_mc)/TMath::Pi();
      break;
   case 5:
      if (TMath::IsNaN(p_mb)) m_logger << kFATAL << "<QPoleMass::MSMassToPoleMass> p_mb is NaN !" << GEndl;
      As  = GetAlphasRun().EvolveAlphas(p_mb)/TMath::Pi();
      break;
   default:
      m_logger << kFATAL << "<PoleMassToMSMass> Unknown flavour: " << nf << GEndl;
   }  
   
   // eq. (34) from (i)
   return m*( 1.0 + 1.333*As 
              + As*As*( 13.44 - 1.041*nl )
              + As*As*As*( 194.0 - 27.0*nl + 0.653*nl*nl) );
}
