/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : RadiatorFunctions                                                     *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory                                                          *
 *        QCD Radiator Functions                                                  *
 *                                                                                *
 * Sources:                                                                       *
 *      - D. Bardin, G. Passarino, The Standard Model in the Making,              *
 *                                 Oxford (1999), Page 503-505                    *
 *                                 and references therein                         *
 *                                                                                *
 *      - P.A. Baikov, K.G. Chetyrkin, J.H. Kuhn,                                 *
 *                 Phys.Rev.Lett.101:012002,2008, (arXiv:0801.1821)               *
 *                                                                                *
 *      - P.A. Baikov, K.G. Chetyrkin, J.H. Kuhn, J. Rittinger                    *
 *                (arXiv:1201.5804)                                               *
 *                                                                                *
 *      - See also: D. Bardin, M. Bilenky, P. Christova, M. Jack,                 *
 *                  L. Kalinovskaya, A. Olchevski, S. Riemann, T. Riemann         *
 *                  CPC 133 (2001) 229-395, hep-ph/9908433                        *
 *                                                                                *
 * see corresponding .h file for author and license information                   *     
 *                                                                                *
 **********************************************************************************/
#include "Gfitter/GMath.h"
#include "Gfitter/GConstants.h"

#include "Gfitter/GParameterRef.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GVariable.h"

#include "GEW/RadiatorFunctions.h"
#include "GEW/DAlphaQED.h"
#include "GEW/AlphaQCDAtQ.h"
#include "GEW/QMassRunning.h"
#include "GEW/RunningAlphaQCD.h"

using namespace Gfitter;

ClassImp(GEW::RadiatorFunctions)

GEW::RadiatorFunctions::RadiatorFunctions()
   : Gfitter::GAuxTheory(),
   m_useNLOAlphasOnly ( kFALSE ),
   m_isUpToDate_Update( kFALSE )   
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   // for error estimation (not used in normal fit)
   if (gStore()->ExistVariable( "GEWFlags::UseNLOAlphasOnlyInRadFun" )) {
      m_useNLOAlphasOnly = gStore()->GetVariable( "GEWFlags::UseNLOAlphasOnlyInRadFun" )->GetBoolValue();
   }

   BookParameter( "mt",                        &p_mt );   
   BookParameter( "MZ",                        &p_MZ );   
   BookParameter( "DeltaAlphasTheoC05_Scale",  &p_DeltaAlphasTheoC05_Scale );
   BookParameter( "DeltaAlphasTheoCMt4_Scale", &p_DeltaAlphasTheoCMt4_Scale );
   //BookParameter( "DeltaAlphasTheoMassive_Scale",   &p_DeltaAlphasTheoMassive_Scale); // negligible
 
   BookTheory   ( "GEW::DAlphaQED",            &t_DAlphaQED );
   BookTheory   ( "GEW::AlphaQCDAtQ/MZ",       &t_AlphasMZ ); 
   BookTheory   ( "GEW::QMassRunning",         &t_qMassRun );
}

void GEW::RadiatorFunctions::Initialise()
{
   // number of quark flavours
   Int_t nf = 5;
   
   // coefficients from Bardin+ Passarino, Page 504
   
   // massless non-singlet corrections (Bardin+Passarino, eq 12.37)  
   m_C02   = 365/24.0 - 11*GMath::Zeta3() + (-11/12.0 + 2/3.0*GMath::Zeta3())*nf;
   m_C03   = 87029/288.0 - 121/8.0*GMath::Zeta2() - 1103/4.0*GMath::Zeta3() + 275/6.0*GMath::Zeta5()  + (-7847/216.0 + 11/6.0*GMath::Zeta2() 
                                   + 262/9.0*GMath::Zeta3() - 25/9.0*GMath::Zeta5())*nf 
      + (151/162.0 - 1/18.0*GMath::Zeta2() - 19/27.0*GMath::Zeta3())*nf*nf;

   // NNNLO QCD adler function (see Baikov et al, arXiv:0801.1821, eq. (5))
   // m_C04   = -156.61 + 18.77*nf -0.7974*nf*nf + 0.0215*nf*nf*nf; // initialized below, since it's varied for theory uncertainty

   // Quadratic massive corrections (Bardin+Passarino, eq 12.38)
   m_C23   = -80.0 + 60*GMath::Zeta3() + (32/9.0 - 8/3.0*GMath::Zeta3())*nf;
   m_CV21  = 12.0;
   m_CV22  = 253/2.0 - 13/3.0*nf;
   m_CV23  = 2522.0 - 855/2.0*GMath::Zeta2() + 310/3.0*GMath::Zeta3() - 5225/6.0*GMath::Zeta5()
      + (-4942/27.0 + 34*GMath::Zeta2() - 394/27.0*GMath::Zeta3() + 1045/27.0*GMath::Zeta5())*nf + (125/54.0 - 2/3.0*GMath::Zeta2())*nf*nf;
   m_CA20  = -6.0;
   m_CA21  = -22.0;
   m_CA22  = -8221/24.0 + 57*GMath::Zeta2() + 117*GMath::Zeta3() + (151/12.0 - 2.0*GMath::Zeta2() - 4.0*GMath::Zeta3())*nf;
   m_CA23  = - 4544045/864.0 + 1340.0*GMath::Zeta2() + 118915/36.0*GMath::Zeta3() - 127.0*GMath::Zeta5() + (71621/162.0 - 209/2.0*GMath::Zeta2() 
             - 216.0*GMath::Zeta3() + 5.0*GMath::Zeta4() + 55.0*GMath::Zeta5())*nf + (-13171/1944.0 + 16/9.0*GMath::Zeta2() + 26/9.0*GMath::Zeta3())*nf*nf;

   // Quartic massive corrections (Bardin+Passarino, eq 12.39)
   m_C42   = 13/3.0 - 4.0*GMath::Zeta3();
   m_CV40  = -6.0;
   m_CV41  = -22.0;
   m_CV42  = -3029/12.0 + 162.0*GMath::Zeta2() + 112.0*GMath::Zeta3() + (143/18.0 - 4.0*GMath::Zeta2() - 8/3.0*GMath::Zeta3())*nf;
   m_CVL42 = -11/2.0 + 1/3.0*nf;
   m_CA40  = 6.0;
   m_CA41  = 10.0;
   m_CA42  = 3389/12.0 - 162.0*GMath::Zeta2() - 220.0*GMath::Zeta3() + (-41/6.0 + 4.0*GMath::Zeta2() + 16/3.0*GMath::Zeta3())*nf;
   m_CAL42 = 77/2.0 - 7/3.0*nf;

}
   

void GEW::RadiatorFunctions::UpdateLocalFlags( GReference& /* ref */ )
{
   m_isUpToDate_Update = kFALSE;
}

void GEW::RadiatorFunctions::Update()
{
   if (m_isUpToDate_Update) return;

   // now, it is uptodate (I mean... it will be)
   m_isUpToDate_Update = kTRUE;

   Double_t pi    = TMath::Pi();
      
   m_mc  = GetQMassRun().GetRunningQuarkMass( GTypes::kCharm );
   m_mb  = GetQMassRun().GetRunningQuarkMass( GTypes::kBottom );

   m_asMZpi    = GetAlphasMZ()/pi;
   m_aQEDMZpi  = GetDAlphaQED().AlphaQEDMZFull() /pi;

   m_RatioMcMZ = GMath::IPow( m_mc/p_MZ,2);
   m_RatioMbMZ = GMath::IPow( m_mb/p_MZ,2);
   m_RatioMZMt = GMath::IPow( p_MZ/p_mt,2);
   
   // common terms in eq. (12.35) and (12.36) of Bardin+Passarino 
   m_UnivQuadratic = ( m_RatioMcMZ + m_RatioMbMZ )*m_C23*GMath::IPow(m_asMZpi,3);
   m_UnivQuartic   = m_RatioMcMZ*m_RatioMcMZ*(m_C42 - TMath::Log(m_RatioMcMZ))*m_asMZpi*m_asMZpi
      + m_RatioMbMZ*m_RatioMbMZ*(m_C42 - TMath::Log(m_RatioMbMZ))*m_asMZpi*m_asMZpi;

   // Power suppressed t-mass correction (Bardin+Passarino, eq 12.40)
   m_Ct2           = m_RatioMZMt*(44/675.0 - 2/135.0*TMath::Log(m_RatioMZMt)); 

   // Singlet axial corrections (Bardin+Passarino, eq 12.41)
   m_I_2          = -37/12.0 + TMath::Log(m_RatioMZMt) + 7/81.0*m_RatioMZMt + 0.0132*m_RatioMZMt*m_RatioMZMt;
   m_I_3          = -5075/216.0 + 23/6.0*GMath::Zeta2() + GMath::Zeta3()
      + 67/18.0*TMath::Log(m_RatioMZMt) + 23/12.0*GMath::IPow(TMath::Log(m_RatioMZMt),2);
   
   // rough estimate from geometric series: C05 = C04*(C04/C03)
   //m_C05   = m_C04/m_C03*m_C04*(1 - p_DeltaAlphasTheoC05_Scale);
   //New: more conservative error estimate: take full NNNLO result as theoretical uncertainty
   Int_t nf = 5;
   m_C04 = (-156.61 + 18.77*nf -0.7974*nf*nf + 0.0215*nf*nf*nf)*(p_DeltaAlphasTheoC05_Scale);
   m_C05 = 0.;

   // Singlet axial corrections
   // NNNLO QCD correction (see Baikov et al, arXiv:1201.5804, eq. (3))
   m_I_4          = 49.0309 - 17.6637*TMath::Log(m_RatioMZMt) + 14.6597*GMath::IPow(TMath::Log(m_RatioMZMt),2) 
      + 3.6736*GMath::IPow(TMath::Log(m_RatioMZMt),3);

   // Conservative estimate: take the full NNNLO result as theoretical uncertainty
   m_I_4 *= (p_DeltaAlphasTheoCMt4_Scale);
   m_I_5 = 0;

   // rough estimate from geometric series
   //m_I_5 = m_I_4/m_I_3*m_I_4*(1.0 - p_DeltaAlphasTheoCMt4_Scale);

   // vector singlet corrections (only used for full hadronic width, cannot be assigned to one single flavor)
   // see for example Bardin+Passarino, eq. 12.17 and eq. 12.42
   // for NNNLO update: Baikov et al, arXiv:1201.5804, eq. (2) and eq. (3)
   m_rVS   = -0.4132*GMath::IPow(m_asMZpi,3) - 4.9841*GMath::IPow(m_asMZpi,4);
   // theoretical error on this is neglible (estimated by conservatively dropping the alphas^4 term or taking it times 2)

   // possible: rough estimate from geometric series 
   // -> (but even more negligible)
   //m_rVS += -4.9841/0.4132*4.9841*GMath::IPow(m_asMZpi,5)*(1.0 - p_DeltaAlphasTheoCMt4_Scale);

   SetUpToDate();
}

// (Bardin+Passarino, eq 12.35)
Double_t GEW::RadiatorFunctions::GetRVq( GTypes::Particle ParticleType, Double_t Charge )
{   
   Update();
   
   Double_t mquark  = 0;
   Double_t mqprime = 0;
   Double_t mq6Term = 0;
   
   // corrections arising only for c and b
   if( ParticleType == GTypes::kCharm ){
      mquark  = m_RatioMcMZ;
      mqprime = m_RatioMbMZ;
      mq6Term = GMath::IPow(mquark,3)*(8.0 + 16/27.0*(155.0 + 6.0*TMath::Log(mquark))*m_asMZpi);
   }
   else if( ParticleType == GTypes::kBottom){
      mquark  = m_RatioMbMZ;
      mqprime = m_RatioMcMZ;
      mq6Term = GMath::IPow(mquark,3)*(8.0 + 16/27.0*(155.0 + 6.0*TMath::Log(mquark))*m_asMZpi);
   }
   
   // terms with mq^2 (zero for light flavors), 
   // obvious typo in last exponential of Bardin+Passarino corrected (2->3) 
   Double_t mq2Terms = mquark*(m_CV21*m_asMZpi + m_CV22*GMath::IPow(m_asMZpi,2)
                                + (m_useNLOAlphasOnly ? 0 : m_CV23*GMath::IPow(m_asMZpi,3)) );
   
   // terms with mq^4 (zero for light flavors)
   Double_t mq4Terms = mquark==0 ? 0 : GMath::IPow(mquark, 2)*
      ( m_CV41*m_asMZpi + (m_CV42 + m_CVL42*TMath::Log(mquark))*GMath::IPow(m_asMZpi,2) );

   // theory uncertainty due to higher orders missing for the massive terms
   // -> found to be negligible compared to others
   //mq2Terms += mquark*m_CV23*GMath::IPow(m_asMZpi,3)*(1.0-p_DeltaAlphasTheoMassive_Scale);
   //if (mquark!=0){
   //  mq4Terms += GMath::IPow(mquark,2)*(m_CV42 + m_CVL42*TMath::Log(mquark))*GMath::IPow(m_asMZpi,2)*(1.0-p_DeltaAlphasTheoMassive_Scale);
   //}

   Double_t RadV = ( 1.0 + 3/4.0*Charge*Charge*m_aQEDMZpi + m_asMZpi - 1/4.0*Charge*Charge*m_aQEDMZpi*m_asMZpi
                     + (m_C02 + m_Ct2)*GMath::IPow(m_asMZpi,2) 
                     + (m_useNLOAlphasOnly ? 0 : ( m_C03*GMath::IPow(m_asMZpi,3) + 
                                                   m_C04*GMath::IPow(m_asMZpi,4) +
                                                   m_C05*GMath::IPow(m_asMZpi,5) ))
                     + m_UnivQuadratic + mq2Terms + m_UnivQuartic + mq4Terms
                     + 12.0*GMath::IPow(mqprime*m_asMZpi,2) - mq6Term );

   return RadV;
}
  
// (Bardin+Passarino, eq 12.36)
Double_t GEW::RadiatorFunctions::GetRAq( GTypes::Particle ParticleType, Double_t Charge, Double_t T3 )
{
   Update();
   
   
   Double_t mquark   = 0;
   Double_t mqprime  = 0;
   Double_t mqmtTerm = 0;

   // corrections arising only for c and b
   if( ParticleType == GTypes::kCharm ){
      mquark   = m_RatioMcMZ;
      mqprime  = m_RatioMbMZ;
      mqmtTerm = -10.0*GMath::IPow( m_mc/p_mt,2)*(8/81.0 + 1/54.0*TMath::Log(m_RatioMZMt))*m_asMZpi*m_asMZpi;
   }
   else if( ParticleType == GTypes::kBottom){
      mquark   = m_RatioMbMZ;
      mqprime  = m_RatioMcMZ;
      mqmtTerm = -10.0*GMath::IPow( m_mb/p_mt,2)*(8/81.0 + 1/54.0*TMath::Log(m_RatioMZMt))*m_asMZpi*m_asMZpi;
   }

   // terms with mq^2 (zero for light flavors)
   Double_t mq2Terms = mquark*( m_CA20 + m_CA21*m_asMZpi + m_CA22*GMath::IPow(m_asMZpi,2) 
                                + 6.0*(3.0 + TMath::Log(m_RatioMZMt))*GMath::IPow(m_asMZpi,2) 
                                + (m_useNLOAlphasOnly ? 0 : m_CA23*GMath::IPow(m_asMZpi,3)) );

   // terms with mq^4 (zero for light flavors)
   Double_t mq4Terms = mquark==0 ? 0 : GMath::IPow(mquark, 2)* ( m_CA40 + m_CA41*m_asMZpi 
                                                                 + (m_CA42 + m_CAL42*TMath::Log(mquark))*GMath::IPow(m_asMZpi,2) );

   // theory uncertainty due to higher orders missing for the massive terms
   // -> found to be negligible compared to others
   //mq2Terms += mquark*m_CA23*GMath::IPow(m_asMZpi,3)*(1.0-p_DeltaAlphasTheoMassive_Scale);
   //if (mquark!=0){
   // mq4Terms += GMath::IPow(mquark,2)*(m_CA42 + m_CAL42*TMath::Log(mquark))*GMath::IPow(m_asMZpi,2)*(1.0-p_DeltaAlphasTheoMassive_Scale);
   //}
     
   Double_t RadA = ( 1.0 + 3/4.0*Charge*Charge*m_aQEDMZpi + m_asMZpi - 1/4.0*Charge*Charge*m_aQEDMZpi*m_asMZpi
                     + (m_C02 + m_Ct2 - 2.0*T3*m_I_2)*GMath::IPow(m_asMZpi,2) 
                     + (m_useNLOAlphasOnly ? 0 : ( (m_C03 - 2.0*T3*m_I_3)*GMath::IPow(m_asMZpi,3) + 
                                                   (m_C04 - 2.0*T3*m_I_4)*GMath::IPow(m_asMZpi,4) +
                                                   (m_C05 - 2.0*T3*m_I_5)*GMath::IPow(m_asMZpi,5) ))
                     + m_UnivQuadratic + mq2Terms + mqmtTerm + m_UnivQuartic + mq4Terms
                     -12.0*mqprime*mqprime*GMath::IPow(m_asMZpi,2) );

   return RadA;
}
