/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : Sigma0had                                                             *
 *                                                                                *
 * Description:                                                                   *
 *      Prediction of the total hadronic cross section                            *
 *      (parametrised formula for two-loop result)                                *
 *                                                                                *
 * Papers:                                                                        *
 *     (i) Ayres Freitas, arxiv.org/abs/1401.2447                                      *
 *         (need to check that this is the final version                          *
 *                                                                                *
 * see corresponding .h file for author and license information                   *     
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GConstants.h"

#include "GEW/Sigma0hadParametrisation.h"

using namespace Gfitter;

ClassImp(GEW::Sigma0hadParametrisation)

GEW::Sigma0hadParametrisation::Sigma0hadParametrisation() : 
   Gfitter::GAuxTheory()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   const TString& logMH = gStore()->GetVariable( "GEWFlags::logMH" )->GetStringValue();
   m_logger << kINFO << "Using logMH: \"" << logMH << "\"" << GEndl;
   
   if      (logMH == "Yes" ) m_logMH  = kTRUE;
   else if (logMH == "No" )  m_logMH  = kFALSE;
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::logMH\": \"" << logMH << "\""
               << ". Possible are: \"Yes\" and \"No\"\""
               << GEndl;
   }
   
   BookParameter( "MZ",                   &p_MZ );   
   BookParameter( "mt",                   &p_mt );   
   BookParameter( "DeltaSigma0had_Scale", &p_DeltaSigma0had_Scale );   
   BookTheory   ( "GEW::AlphaQCDAtQ/MZ",  &t_AlphasMZ );   
   BookTheory   ( "GEW::DAlphaQED",       &t_DAlphaQED );   
   BookTheory   ( "GEW::MH",              &t_MH );
}


// initialise coefficienct
// eq. (12) of (i) 
void GEW::Sigma0hadParametrisation::Initialise()
{
  //From A.Freitas http://arxiv.org/abs/1401.2447 Table 7
   m_X0  =  41488.4;
   m_c1  =  3.88;
   m_c2  =  0.829;
   m_c3  =  -0.911;
   m_c4  =  0.0076;
   m_c5  =  61.10;
   m_c6  =  16.;
   m_c7  =  -2.0;
   m_c8  =  -0.59;
   m_c9  =  -579.4;
   m_c10 =  38.;
   m_c11 =  -0.26;
   m_c12 =  6.5;
   m_c13 =  84.;
   m_c14 =  9.5;
   m_c15 =  -86152.;
}

// parametrised formula (10) of ref (i)
Double_t GEW::Sigma0hadParametrisation::GetSigma0had()
{   

   Double_t sigma0had = 0;

   Double_t MH      = TMath::Abs( GetMH().GetValue() );
   if( m_logMH ) MH = TMath::Exp( GetMH().GetValue() ); 
      
   Double_t m_LH          = TMath::Log(MH/125.7);
   Double_t m_dH          = (MH/125.7) - 1.0;
   Double_t m_dZ          = (p_MZ/91.1876) - 1.0;
   Double_t m_dts         = TMath::Power((p_mt/173.2),2) - 1.0;
 
   Double_t m_deAlpha     = (GetDAlphaQED().DAlphaQEDMZ()/0.05900) - 1.0;
   Double_t m_deAlphaS    = (GetAlphasMZ()/0.1184) - 1.0;  

   sigma0had = m_X0 
     + m_c1 * m_LH 
     + m_c2 * m_LH * m_LH 
     + m_c3 * m_dH 
     + m_c4 * m_dH * m_dH
     + m_c5 * m_dts
     + m_c6 * m_dts * m_dts
     + m_c7 * m_dts * m_LH
     + m_c8 * m_dts * m_LH * m_LH
     + m_c9 * m_deAlphaS
     + m_c10 * m_deAlphaS * m_deAlphaS
     + m_c11 * m_deAlphaS * m_LH
     + m_c12 * m_deAlphaS * m_dts
     + m_c13 * m_deAlpha
     + m_c14 * m_deAlpha * m_LH
     + m_c15 * m_dZ;
   
   /*   sigma0had = m_X0 
     + m_c1*deAlpha 
     + m_c2*deAlphaS 
     + m_c3*deAlphaS*deAlphaS 
     + m_c4*dZ
     + m_c5*TMath::Exp(LH)
     + m_c6*TMath::Exp(2*LH)
     + m_c7*LH
     + m_c8*deAlpha*LH
     + m_c9*deAlphaS*LH
     + m_c10*LH*LH
     + m_c11*dts
     + m_c12*deAlphaS*dts
     + m_c13*LH*dts
     + m_c14*LH*LH*dts
     + m_c15*dts*dts;
   */

   sigma0had += (1.0-p_DeltaSigma0had_Scale)*6.0; // theoretical uncertainty of 6 pb

   return sigma0had;

}
