/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : Sin2TheatF                                                            *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory for the effective weak mixing angle (included 2 loops)   *
 *                                                                                *
 * Papers:                                                                        *
 *     (i)  M. Awramik, M. Czakon and A. Freitas                                  *
 *          JHEP 0611, 048 (2006), [hep-ph/0608099]                               *  
 *     (ii) M. Awramik, M. Czakon, A. Freitas and B. A. Kniehl                    *
 *          Nucl. Phys. B 813, 174 (2009), [arXiv:0811.1364 [hep-ph]]             *
 *                                                                                *
 * see corresponding .h file for author and license information                   *     
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GConstants.h"

#include "GEW/Sin2ThetaF.h"
#include "GEW/DAlphaQED.h"
#include "GEW/AlphaQCDAtQ.h"
#include "GEW/MH.h"

using namespace Gfitter;

ClassImp(GEW::Sin2ThetaF)

GEW::Sin2ThetaF::Sin2ThetaF()
   : Gfitter::GAuxTheory()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );
   
   const TString& logMH = gStore()->GetVariable( "GEWFlags::logMH" )->GetStringValue();
   m_logger << kINFO << "Using logMH: \"" << logMH << "\"" << GEndl;
   
   if      (logMH == "Yes" ) m_logMH  = kTRUE;
   else if (logMH == "No" )  m_logMH  = kFALSE;
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::logMH\": \"" << logMH << "\""
               << ". Possible are: \"Yes\" and \"No\"\""
               << GEndl;
   }

   BookParameter( "MZ",                      &p_MZ );    
   BookParameter( "mt",                      &p_mt );   
   BookParameter( "GF",                      &p_GF );   
   BookParameter( "DeltaSin2ThetaF_Scale",     &p_DeltaSin2ThetaF_Scale );   

   BookTheory   ( "GEW::AlphaQCDAtQ/MZ",     &t_AlphasMZ );
   BookTheory   ( "GEW::DAlphaQED",          &t_DAlphaQED ); 
   BookTheory   ( "GEW::MH" , & t_MH );
}

// coefficients of the fitting formulaes 
void GEW::Sin2ThetaF::Initialise()
{
   // table 5 from (i)
   m_s0[0]  = 0.2312527;
   m_s0[1]  = 0.2308772;
   m_s0[2]  = 0.2311395;
   m_s0[3]  = 0.2310286;
   m_d1[0]  = 4.729e-4;
   m_d1[1]  = 4.713e-4;
   m_d1[2]  = 4.726e-4;
   m_d1[3]  = 4.720e-4;
   m_d2[0]  = 2.07e-5;
   m_d2[1]  = 2.05e-5;
   m_d2[2]  = 2.07e-5;
   m_d2[3]  = 2.06e-5;
   m_d3[0]  = 3.85e-6;
   m_d3[1]  = 3.85e-6;
   m_d3[2]  = 3.85e-6;
   m_d3[3]  = 3.85e-6;
   m_d4[0]  = -1.85e-6;
   m_d4[1]  = -1.85e-6;
   m_d4[2]  = -1.85e-6;
   m_d4[3]  = -1.85e-6;
   m_d5[0]  = 2.07e-2;
   m_d5[1]  = 2.06e-2;
   m_d5[2]  = 2.07e-2;
   m_d5[3]  = 2.07e-2;
   m_d6[0]  = -2.851e-3;
   m_d6[1]  = -2.850e-3;
   m_d6[2]  = -2.853e-3; 
   m_d6[3]  = -2.848e-3;
   m_d7[0]  = 1.82e-4;
   m_d7[1]  = 1.82e-4;
   m_d7[2]  = 1.83e-4;
   m_d7[3]  = 1.81e-4;
   m_d8[0]  = -9.74e-6;
   m_d8[1]  = -9.71e-6;
   m_d8[2]  = -9.73e-6;
   m_d8[3]  = -9.73e-6;
   m_d9[0]  = 3.98e-4;
   m_d9[1]  = 3.96e-4;
   m_d9[2]  = 3.98e-4;
   m_d9[3]  = 3.97e-4;
   m_d10[0] = -6.55e-1;
   m_d10[1] = -6.54e-1;
   m_d10[2] = -6.55e-1;
   m_d10[3] = -6.55e-1;

   // eq. (20) from (ii)
   m_s0[4]  =  2.327580e-1;
   m_d1[4]  =  4.749e-4;
   m_d2[4]  =  2.03e-5;
   m_d3[4]  =  3.94e-6;
   m_d4[4]  = -1.84e-6;
   m_d5[4]  =  2.08e-2;
   m_d6[4]  = -9.93e-4;
   m_d7[4]  =  7.08e-5;
   m_d8[4]  = -7.61e-6;
   m_d9[4]  =  4.03e-4;
   m_d10[4] =  6.61e-1;
}

// eff. ew mixing angle
// formulaes (48), (49) from (i) and (18) from (ii)
Double_t GEW::Sin2ThetaF::GetSin2ThetaF( GTypes::Particle ParticleType )
{
   Double_t MH        = GetMH().GetValue(); 
   if( m_logMH ) MH = TMath::Exp( GetMH().GetValue() ); 

   Double_t LogH       = TMath::Log(MH/100);
   Double_t dH         = (MH/100.0);
   Double_t deAlpha    = (GetDAlphaQED().DAlphaQEDMZ()/0.05907) - 1.0;
   Double_t dt         = ((p_mt/178.0)*(p_mt/178.0) - 1.0);
   Double_t deAlphaS   = (GetAlphasMZ()/0.117) - 1.0;   
   Double_t dZ         = (p_MZ/91.1876) - 1.0;

   Double_t sin2ThetaF = 0;

   Int_t index = 0;
   
   if ( ParticleType == GTypes::kElectron || ParticleType == GTypes::kMuon || ParticleType == GTypes::kTau ) 
      index = 0; // charged leptons
   else if ( ParticleType == GTypes::kNeutrino ) {
      index = 1; // neutrinos
   }
   else if ( ParticleType == GTypes::kUp || ParticleType == GTypes::kCharm ) {
      index = 2; // up, charm
   }
   else if ( ParticleType == GTypes::kDown || ParticleType == GTypes::kStrange ) {
      index = 3; // down, strange
   }
   else if ( ParticleType == GTypes::kBottom ) {
      index = 4; // bottom
   }
   else m_logger << kFATAL << "<SIN2ThethaL> Unknown flavour: " << GTypes::GetName(ParticleType) << GEndl;

   bool print = false;
   if (print){
     std::cout << "MH = " << GetMH().GetValue() << std::endl
	       << "MZ = " << p_MZ << std::endl
	       << "mt = " << p_mt << std::endl
	       << "alphas = " << GetAlphasMZ()+0.0 << std::endl
	       << "alpha = " << GetDAlphaQED().DAlphaQEDMZ() << std::endl
	       << "index = " << index << " (0 = charged leptons, 1 = neutrinos)" << std::endl;
   }

   
   sin2ThetaF = ( m_s0[index] + m_d1[index]*LogH + m_d2[index]*LogH*LogH 
                  + m_d3[index]*Gfitter::GMath::IPow(LogH,4) + m_d4[index]*(dH*dH-1) 
                  + m_d5[index]*deAlpha + m_d6[index]*dt + m_d7[index]*dt*dt 
                  + m_d8[index]*dt*(dH-1) + m_d9[index]*deAlphaS + m_d10[index]*dZ );
   
   if (print) std::cout << "sin2theta = " << sin2ThetaF << std::endl;

   // theretical uncertainty for eff. weak mixing angle (please consult (i))
   sin2ThetaF = sin2ThetaF + (1.0 - p_DeltaSin2ThetaF_Scale)*4.7e-5;
   
   if (print) std::cout << "sin2theta with theo uncertainty = " << sin2ThetaF << std::endl << std::endl;
      
   // inform the base class that calculations for this set of parameters the 
   // theory predictions are up-to-date
   // SetUpToDate();

   // test GF dependency of GF
   if( false ){
      Double_t MZ2 = p_MZ*p_MZ;

      Double_t deltaSin2 = ( 1./2.*( TMath::Sqrt(1-TMath::Sqrt(8.)*TMath::Pi()*GConstants::alphaQED()/(MZ2*GConstants::GF())) 
                                     - TMath::Sqrt(1-TMath::Sqrt(8.)*TMath::Pi()*GConstants::alphaQED()/(MZ2*p_GF)) ) );
      sin2ThetaF += deltaSin2;
                            
   } 

   return sin2ThetaF;
}
