/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : Z0Parametrisation                                                     *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory of the parametrisation option                            *
 *      Computes effective weak mixing angle and Partial Z widths                 *
 *                                                                                *
 * Sources:                                                                       *
 *      (i)   G.-C. Cho, K. Hagiwara, Y. Matsumoto and D. Nomura                  *
 *            JHEP 1111 (2011) 068, [arXiv:1104.1769 [hep-ph]]                    *
 *      (ii)  K. Hagiwara, S. Matsumoto, D. Haidt and C. S. Kim                   *
 *            Z. Phys. C 64 (1994) 559, [hep-ph/9409380]                          *
 *      (iii) D. Bardin, G. Passarino, The Standard Model in the Making,          *
 *            Oxford (1999), Page 496- , and references therein                   *
 *      (iv)  P.A. Baikov, K.G. Chetyrkin, J.H. Kuhn, J. Rittinger                *
 *            (arXiv:1201.5804)                                                   *
 *      (v)   A. Freitas: http://arxiv.org/abs/1401.2447
 *                                                                                *
 * see corresponding .h file for author and license information                   *     
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GConstants.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"

#include "GEW/Z0Parametrisation.h"
#include "GEW/Sin2ThetaF.h"
#include "GEW/RadiatorFunctions.h"
#include "GEW/MH.h"

using namespace std;
using namespace Gfitter;
using namespace Gfitter::GTypes;

ClassImp(GEW::Z0Parametrisation)

GEW::Z0Parametrisation::Z0Parametrisation()
  : Z0Base(),
    m_fullTwoLoop( kTRUE ),
    m_isUpToDate_Update( kFALSE )
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   m_fullTwoLoop = kTRUE ;
   if ( gStore()->ExistVariable( "GEWFlags::FullTwoLoop" ) ) {
     m_fullTwoLoop = gStore()->GetVariable( "GEWFlags::FullTwoLoop" )->GetBoolValue();
   }
   m_logger << kINFO << "Using full 2-loop fermionic parametrization: " << m_fullTwoLoop << GEndl;


   const TString& logMH = gStore()->GetVariable( "GEWFlags::logMH" )->GetStringValue();
   m_logger << kINFO << "Using logMH: \"" << logMH << "\"" << GEndl;

   if      (logMH == "Yes" ) m_logMH  = kTRUE;
   else if (logMH == "No" )  m_logMH  = kFALSE;
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::logMH\": \"" << logMH << "\""
               << ". Possible are: \"Yes\" and \"No\"\""
               << GEndl;
   }

   BookTheory( "GEW::RadiatorFunctions" , &t_radFun );
   BookTheory( "GEW::MH",                 &t_MH );
   BookParameter( "mt",                   &p_mt );
   BookTheory   ( "GEW::AlphaQCDAtQ/MZ",  &t_AlphasMZ );
   BookTheory   ( "GEW::DAlphaQED",       &t_DAlphaQED );
   BookParameter( "MZ",                   &p_MZ );

   BookParameter( "DeltaGammaZlept_Scale", &p_DeltaGammaZlept_Scale); 
   BookParameter( "DeltaGammaZnu_Scale",   &p_DeltaGammaZnu_Scale); 
   BookParameter( "DeltaGammaZds_Scale",   &p_DeltaGammaZds_Scale);	 
   BookParameter( "DeltaGammaZuc_Scale",   &p_DeltaGammaZuc_Scale);	 
   BookParameter( "DeltaGammaZb_Scale",    &p_DeltaGammaZb_Scale);    
}

void GEW::Z0Parametrisation::UpdateLocalFlags( GReference& /* ref */ )
{
   m_isUpToDate_Update = kFALSE;
}

void GEW::Z0Parametrisation::Update()
{
   if (m_isUpToDate_Update) return;

   Double_t MH        = GetMH().GetValue(); 
   if (m_logMH) MH = TMath::Exp( GetMH().GetValue() );

   // eq. 24 from (i)
   Double_t xt = (p_mt - 172.)/3.;
   Double_t xh = TMath::Log( MH/100. )/TMath::Log( 10. );
   Double_t xs = (GetAlphasMZ() - 0.118)/0.003;
   // eq. 15 from (i)
   Double_t xalpha = (GetDAlphaQED().DAlphaHad5() - 0.0277)/0.0003;
   
   // eq. 23a and 23b from (i)
   m_Delta_SZ = 0.2217*xh - 0.1188*xh*xh + 0.0320*xh*xh*xh - 0.0014*xt + 0.0005*xs;
   m_Delta_TZ = (-0.0995*xh - 0.2858*xh*xh + 0.1175*xh*xh*xh + 0.0367*xt + 0.00026*xt*xt 
		 -0.0017*xh*xt - 0.0033*xs - 0.0001*xt*xs );
   
   // eq. 13a and 13b from (i)
   Double_t Delta_gbarZ2 = 0.00412*m_Delta_TZ;
   Double_t Delta_sbar2  = 0.00360*m_Delta_SZ - 0.00241*m_Delta_TZ + 0.00011*xalpha;
   
   // eq. 12a-12i from (i)
   m_gLv =  0.50199 + 0.45250*Delta_gbarZ2 + 0.00469*Delta_sbar2;
   m_gRv =  0;
   m_gLe = -0.26920 - 0.24338*Delta_gbarZ2 + 1.00413*Delta_sbar2;
   m_gRe =  0.23207 + 0.20912*Delta_gbarZ2 + 1.00784*Delta_sbar2;
   m_gLu =  0.34675 + 0.31309*Delta_gbarZ2 - 0.66793*Delta_sbar2;
   m_gRu = -0.15470 - 0.13942*Delta_gbarZ2 - 0.67184*Delta_sbar2;
   m_gLd = -0.42434 - 0.38279*Delta_gbarZ2 + 0.33166*Delta_sbar2;
   m_gRd =  0.07734 + 0.06971*Delta_gbarZ2 + 0.33590*Delta_sbar2;
   // plus eq. 25a, 25b from (i)
   m_gLb = -0.42116 - 0.38279*Delta_gbarZ2 + 0.33166*Delta_sbar2 - 0.000058*xh + 0.000128*xt;
   m_gRb =  0.07742 + 0.06971*Delta_gbarZ2 + 0.33590*Delta_sbar2 - 0.000042*xh - 0.000025*xh*xh*xh*xh; 
   
   // vector radiator functions (not used)
   // table 4 from (i) 
   m_CV[0] = 1;                  // neutrino
   m_CV[1] = 1;                  // electron, muon, tau
   m_CV[2] = 3.1666 + 0.0030*xs; // up
   m_CV[3] = 3.1667 + 0.0030*xs; // down
   m_CV[4] = 3.1667 + 0.0030*xs; // charm
   m_CV[5] = 3.1667 + 0.0030*xs; // strange
   m_CV[6] = 3.1185 + 0.0030*xs; // bottom
   
   // axial radiator functions (not used)
   // table 4 from (i) 
   m_CA[0] = 1;                               // neutrino
   m_CA[1] = 1;                               // electron, muon, tau
   m_CA[2] = 3.1377 + 0.00014*xt + 0.0041*xs; // up
   m_CA[3] = 3.0956 - 0.00015*xt + 0.0019*xs; // down
   m_CA[4] = 3.1369 + 0.00014*xt + 0.0043*xs; // charm
   m_CA[5] = 3.0956 - 0.00015*xt + 0.0019*xs; // strange
   m_CA[6] = 3.0758 - 0.00015*xt + 0.0028*xs; // bottom

    // parametrized full 2-loop fermionic calculation (A. Freitas)
   //if (m_fullTwoLoop) 
   //{ // now always evaluate
   m_LH          = TMath::Log(MH/125.7);
   m_dH          = (MH/125.7) - 1.0;
   m_dZ          = (p_MZ/91.1876) - 1.0;
   m_dts         = TMath::Power((p_mt/173.2),2) - 1.0;
   m_deAlpha     = (GetDAlphaQED().DAlphaQEDMZ()/0.05900) - 1.0;
   m_deAlphaS    = (GetAlphasMZ()/0.1184) - 1.0;  
   
   // now, it is uptodate 
   m_isUpToDate_Update = kTRUE;

   SetUpToDate();
}

// effective weak mixing angle
// see also class Sin2ThetaF.h
Double_t GEW::Z0Parametrisation::GetSin2Eff( GTypes::Particle ParticleType )
{   
   Double_t sineff = 0;

   sineff = GetSinEffF().GetSin2ThetaF( ParticleType );

   return sineff;
}

// relation between left-right and vector coupling
Double_t GEW::Z0Parametrisation::gVf( GTypes::Particle ParticleType, Double_t /*T3*/ )
{
   Update();
   Double_t coupl = 0;

   switch(ParticleType){
   case kElectron:
   case kMuon:
   case kTau:
      coupl = m_gLe + m_gRe;
      break;
   case kNeutrino:
      coupl = m_gLv + m_gRv;
      break;
   case kUp:
   case kCharm:
      coupl = m_gLu + m_gRu;
      break;
   case kDown:
   case kStrange:
      coupl = m_gLd + m_gRd;
      break;
   case kBottom:
      coupl = m_gLb + m_gRb;
      break;
   default:
      m_logger << kFATAL << "False input for ParticleType" << GTypes::GetName(ParticleType) << GEndl;
      break;
   }

   return coupl;
}

// relation between left-right and axial coupling
Double_t GEW::Z0Parametrisation::gAf( GTypes::Particle ParticleType, Double_t /*T3*/ )
{
   Update();
   Double_t coupl = 0;
 
   switch(ParticleType){
   case kElectron:
   case kMuon:
   case kTau:
      coupl  = m_gLe - m_gRe;
      break;
   case kNeutrino:
      coupl  = m_gLv - m_gRv;
      break;
   case kUp:
   case kCharm:
      coupl  = m_gLu - m_gRu;
      break;
   case kDown:
   case kStrange:
      coupl  = m_gLd - m_gRd;
      break;
   case kBottom:
      coupl  = m_gLb - m_gRb;
      break;
   default:
      m_logger << kFATAL << "False input for ParticleType" << GTypes::GetName(ParticleType) << GEndl;
      break;
   }
   
   //
   return coupl;
}

// partial width of Z boson
// eq. 26 from (i) but using different radiator functions includig more radiative corrections
// (more details can be found in the RaditorFunction class)
Double_t GEW::Z0Parametrisation::GammaZff( GTypes::Particle ParticleType, Double_t Charge, Double_t mf )
{
   Update();
   
   Double_t Gamma0 = p_GF*GMath::IPow(p_MZ,3) / (6.0*GMath::Sqrt2()*TMath::Pi());
   Double_t aQEDMZ = GConstants::alphaQED()/( 1 - GetDAlphaQED().DAlphaQEDMZt() );
   Double_t T3     = 0.5*TMath::Sign(1.0,Charge+0.1);
   //Double_t gV     = gVf( ParticleType, T3 );
   Double_t gA     = gAf( ParticleType, T3 );
   Double_t RadV   = 1;
   Double_t RadA   = 1;
   Double_t beta   = 1;
   Double_t ratio  = 1;
   int      index  = 0;

   switch(ParticleType){
   case kNeutrino :
      index  = 0;
      RadV   = 1;
      RadA   = 1;
      break;
   case kElectron :
   case kMuon     :
   case kTau      :
      index  = 1;
      // eq. 3.13a, 3.13b, and 3.14 from (ii)
      ratio  = std::pow( mf/p_MZ, 2 );
      beta   = std::sqrt( 1 - 4*ratio );
      RadV   = beta*(3-beta*beta)/2.*(1.0 + 0.75*aQEDMZ/TMath::Pi()*Charge*Charge);
      RadA   = beta*beta*beta*(1.0 + 0.75*aQEDMZ/TMath::Pi()*Charge*Charge);
      break;
   case kUp       : 
      index  = 2;
      RadV   = 3*GetRadFun().GetRVq( ParticleType, Charge);
      RadA   = 3*GetRadFun().GetRAq( ParticleType, Charge, T3);
      break;
   case kDown     :
      index  = 3;
      RadV   = 3*GetRadFun().GetRVq( ParticleType, Charge);
      RadA   = 3*GetRadFun().GetRAq( ParticleType, Charge, T3);
      break;
   case kCharm    :
      index  = 4;
      RadV   = 3*GetRadFun().GetRVq( ParticleType, Charge);
      RadA   = 3*GetRadFun().GetRAq( ParticleType, Charge, T3);
      break;
   case kStrange  :
      index  = 5;
      RadV   = 3*GetRadFun().GetRVq( ParticleType, Charge);
      RadA   = 3*GetRadFun().GetRAq( ParticleType, Charge, T3);
      break;
   case kBottom   :
      index  = 6;
      RadV   = 3*GetRadFun().GetRVq( ParticleType, Charge);
      RadA   = 3*GetRadFun().GetRAq( ParticleType, Charge, T3);
      break;
   default:
      m_logger << kFATAL << "false input for ParticleType : " << GTypes::GetName(ParticleType) << GEndl;
      break;
   } 

   // instead using gV and gA, take ratio of couplings, since sineff more accurate
   Double_t sineff     = GetSin2Eff(ParticleType);
   Double_t RatioCplgs = 1.0 - 4.0*TMath::Abs(Charge)*sineff;
   Double_t Gamma = Gamma0*( gA*gA*( RatioCplgs*RatioCplgs*RadV + RadA ) + m_ImKappa[index]*RadV ) + m_EWQCD[index];

   return Gamma;
}

// please note: Gamma_had != Gamma_u + Gamma_d + Gamma_c + Gamma_s + Gamma_b (see for instance ref. (iii) Bardin+Passarino on page 500)
// the singlet vector contribution cannot be assigned to single quark flavor (take it only into account for full hadronic width) 
Double_t GEW::Z0Parametrisation::GetGaZhad()
{
   Update();
   
   Double_t ghad   =  GetGaZup() + GetGaZdown() + GetGaZcharm() + GetGaZstrange() + GetGaZbottom();

   GTypes::Particle types[] = { GTypes::kUp, GTypes::kDown, GTypes::kCharm, GTypes::kStrange, GTypes::kBottom };
   Double_t charges[] = { GMath::TwoThird(), -GMath::OneThird(), GMath::TwoThird(), -GMath::OneThird(), -GMath::OneThird() };
   
   // ref. (iii)  Bardin+Passarino, eq. (12.42)
   // and ref. (iv) Baikov et al, eq. (2) and eq. (3)
   Double_t sum = 0;
   for( int i = 0; i < 5; i++ ){
     Double_t T3 = 0.5*TMath::Sign(1.0,charges[i]+0.1);
     Double_t gV = gAf( types[i], T3 ) - 2.0*charges[i]*GetSin2Eff( types[i] ); // gVf( types[i], T3 );
     sum += gV; // note: vf = 2*gVf
   }
   
   Double_t Gamma0 = p_GF*GMath::IPow(p_MZ,3) / (6.0*GMath::Sqrt2()*TMath::Pi());
   
   ghad += Gamma0*3*sum*sum*GetRadFun().GetrVS();

   return ghad;
}

Double_t GEW::Z0Parametrisation::GetGaZelectron()
{
  Double_t res = 0.;
  if (m_fullTwoLoop) {
    res =  GaZelectronParametrisation(); //GammaZff( GTypes::kElectron, -1, GConstants::me() );
  } else {
    res =  GammaZff( GTypes::kElectron, -1, GConstants::me() );
  }
  
  // theory uncertainty
  res += (1.0 - p_DeltaGammaZlept_Scale)*0.012e-3; // theory uncertainty: 0.012 MeV

  return res;
 
}

Double_t GEW::Z0Parametrisation::GetGaZmuon()
{
  Double_t res = 0.;
  if (m_fullTwoLoop) {
    res =  GaZmuonParametrisation();
  } else {
    res =  GammaZff( GTypes::kMuon, -1, GConstants::mmu() );
  }

  // theory uncertainty
  res += (1.0 - p_DeltaGammaZlept_Scale)*0.012e-3; // theory uncertainty: 0.012 MeV

  return res;
 
}

Double_t GEW::Z0Parametrisation::GetGaZtau()
{   
  Double_t res = 0.;
  if (m_fullTwoLoop) {
    res =  GaZtauParametrisation();
  } else {
    res =  GammaZff( GTypes::kTau, -1, GConstants::mtau() );
  }

  // theory uncertainty
  res += (1.0 - p_DeltaGammaZlept_Scale)*0.012e-3; // theory uncertainty: 0.012 MeV

  return res;
 
}

Double_t GEW::Z0Parametrisation::GetGaZneutrino()
{   
  Double_t res = 0.;
  if (m_fullTwoLoop) {
    res =  GaZneutrinoParametrisation();
  } else {
    res =  GammaZff( GTypes::kNeutrino, 0, 0);
  }

  // theory uncertainty
  res += (1.0 - p_DeltaGammaZnu_Scale)*0.014e-3; // theory uncertainty: 0.014 MeV

  return res;
 
}

Double_t GEW::Z0Parametrisation::GetGaZup()
{   
  Double_t res = 0.;
  if (m_fullTwoLoop) {
    res =  GaZupParametrisation(); 
  } else {
    res =  GammaZff( GTypes::kUp, GMath::TwoThird() );
  }

  // theory uncertainty
  res += (1.0 - p_DeltaGammaZuc_Scale)*0.12e-3; // theory uncertainty: 0.12 MeV

  return res;
 
}

Double_t GEW::Z0Parametrisation::GetGaZdown()
{   
  Double_t res = 0.;
  if (m_fullTwoLoop) {
    res =  GaZdownParametrisation(); 
  } else {
    res =  GammaZff( GTypes::kDown, -GMath::OneThird() );
  }

  // theory uncertainty
  res += (1.0 - p_DeltaGammaZds_Scale)*0.09e-3; // theory uncertainty: 0.09 MeV

  return res;
 
}

Double_t GEW::Z0Parametrisation::GetGaZstrange()
{   
  Double_t res = 0.;
  if (m_fullTwoLoop) {
    res =  GaZstrangeParametrisation(); 
  } else {
    res =  GammaZff( GTypes::kStrange, -GMath::OneThird() );
  }

  // theory uncertainty
  res += (1.0 - p_DeltaGammaZds_Scale)*0.09e-3; // theory uncertainty: 0.09 MeV

  return res;
 
}

Double_t GEW::Z0Parametrisation::GetGaZcharm()
{   
  Double_t res = 0.;
  if (m_fullTwoLoop) {
    res =  GaZcharmParametrisation();
  } else {
    res =  GammaZff( GTypes::kCharm, GMath::TwoThird() );
  }

  // theory uncertainty
  res += (1.0 - p_DeltaGammaZuc_Scale)*0.12e-3; // theory uncertainty: 0.12 MeV

  return res;
 
}

Double_t GEW::Z0Parametrisation::GetGaZbottom()
{   
  Double_t res = 0.;
  if (m_fullTwoLoop) {
    res =  GaZbottomParametrisation();
  } else {
    res =  GammaZff( GTypes::kBottom, -GMath::OneThird() );
  }

  // theory uncertainty
  res += (1.0 - p_DeltaGammaZb_Scale)*0.21e-3; // theory uncertainty: 0.21 MeV

  return res;
 
}

// effective couplings / A = axial, V = vector

Double_t GEW::Z0Parametrisation::GetgAeff( Gfitter::GTypes::Particle particle, Double_t T3, Double_t /*ch*/ )
{   
   Double_t gA = gAf( particle, T3 );
  
   return gA;
}    

Double_t GEW::Z0Parametrisation::GetgVeff( Gfitter::GTypes::Particle particle, Double_t T3, Double_t /*ch*/ )
{
   Double_t gV = gVf( particle, T3 );
         
   return gV;
}

Double_t GEW::Z0Parametrisation::GetDeltaS()
{   
   Update();
   return m_Delta_SZ;
}

Double_t GEW::Z0Parametrisation::GetDeltaT()
{   
   Update();
   return m_Delta_TZ;
}


Double_t GEW::Z0Parametrisation::GaZelectronParametrisation()
{
   Update();

   double value=0.0;

   value = m_cEle[0] 
     + m_cEle[1] * m_LH 
     + m_cEle[2] * m_LH * m_LH 
     + m_cEle[3] * m_dH 
     + m_cEle[4] * m_dH * m_dH
     + m_cEle[5] * m_dts
     + m_cEle[6] * m_dts * m_dts
     + m_cEle[7] * m_dts * m_LH
     + m_cEle[8] * m_dts * m_LH * m_LH
     + m_cEle[9] * m_deAlphaS
     + m_cEle[10] * m_deAlphaS * m_deAlphaS
     + m_cEle[11] * m_deAlphaS * m_LH
     + m_cEle[12] * m_deAlphaS * m_dts
     + m_cEle[13] * m_deAlpha
     + m_cEle[14] * m_deAlpha * m_LH
     + m_cEle[15] * m_dZ;
   
   return value / 1000.;
}

Double_t GEW::Z0Parametrisation::GaZmuonParametrisation()
{
   Update();
   
   double value=0.0;
   
   value = m_cMuo[0] 
     + m_cMuo[1] * m_LH 
     + m_cMuo[2] * m_LH * m_LH 
     + m_cMuo[3] * m_dH 
     + m_cMuo[4] * m_dH * m_dH
     + m_cMuo[5] * m_dts
     + m_cMuo[6] * m_dts * m_dts
     + m_cMuo[7] * m_dts * m_LH
     + m_cMuo[8] * m_dts * m_LH * m_LH
     + m_cMuo[9] * m_deAlphaS
     + m_cMuo[10] * m_deAlphaS * m_deAlphaS
     + m_cMuo[11] * m_deAlphaS * m_LH
     + m_cMuo[12] * m_deAlphaS * m_dts
     + m_cMuo[13] * m_deAlpha
     + m_cMuo[14] * m_deAlpha * m_LH
     + m_cMuo[15] * m_dZ;
   
   return value / 1000.;
}

Double_t GEW::Z0Parametrisation::GaZtauParametrisation()
{
   Update();
   
   double value=0.0;
   
   value = m_cTau[0] 
     + m_cTau[1] * m_LH 
     + m_cTau[2] * m_LH * m_LH 
     + m_cTau[3] * m_dH 
     + m_cTau[4] * m_dH * m_dH
     + m_cTau[5] * m_dts
     + m_cTau[6] * m_dts * m_dts
     + m_cTau[7] * m_dts * m_LH
     + m_cTau[8] * m_dts * m_LH * m_LH
     + m_cTau[9] * m_deAlphaS
     + m_cTau[10] * m_deAlphaS * m_deAlphaS
     + m_cTau[11] * m_deAlphaS * m_LH
     + m_cTau[12] * m_deAlphaS * m_dts
     + m_cTau[13] * m_deAlpha
     + m_cTau[14] * m_deAlpha * m_LH
     + m_cTau[15] * m_dZ;
   
   return value / 1000.;
}

Double_t GEW::Z0Parametrisation::GaZneutrinoParametrisation()
{
   Update();
   
   double value=0.0;
   
   value = m_cNeu[0] 
     + m_cNeu[1] * m_LH 
     + m_cNeu[2] * m_LH * m_LH 
     + m_cNeu[3] * m_dH 
     + m_cNeu[4] * m_dH * m_dH
     + m_cNeu[5] * m_dts
     + m_cNeu[6] * m_dts * m_dts
     + m_cNeu[7] * m_dts * m_LH
     + m_cNeu[8] * m_dts * m_LH * m_LH
     + m_cNeu[9] * m_deAlphaS
     + m_cNeu[10] * m_deAlphaS * m_deAlphaS
     + m_cNeu[11] * m_deAlphaS * m_LH
     + m_cNeu[12] * m_deAlphaS * m_dts
     + m_cNeu[13] * m_deAlpha
     + m_cNeu[14] * m_deAlpha * m_LH
     + m_cNeu[15] * m_dZ;
   
   return value / 1000.;
}

Double_t GEW::Z0Parametrisation::GaZupParametrisation()
{
   Update();
   
   double value=0.0;
   
   value = m_cUp[0] 
     + m_cUp[1] * m_LH 
     + m_cUp[2] * m_LH * m_LH 
     + m_cUp[3] * m_dH 
     + m_cUp[4] * m_dH * m_dH
     + m_cUp[5] * m_dts
     + m_cUp[6] * m_dts * m_dts
     + m_cUp[7] * m_dts * m_LH
     + m_cUp[8] * m_dts * m_LH * m_LH
     + m_cUp[9] * m_deAlphaS
     + m_cUp[10] * m_deAlphaS * m_deAlphaS
     + m_cUp[11] * m_deAlphaS * m_LH
     + m_cUp[12] * m_deAlphaS * m_dts
     + m_cUp[13] * m_deAlpha
     + m_cUp[14] * m_deAlpha * m_LH
     + m_cUp[15] * m_dZ;
   
   return value / 1000.;
}

Double_t GEW::Z0Parametrisation::GaZdownParametrisation()
{
   Update();
   
   double value=0.0;
   
   value = m_cDow[0] 
     + m_cDow[1] * m_LH 
     + m_cDow[2] * m_LH * m_LH 
     + m_cDow[3] * m_dH 
     + m_cDow[4] * m_dH * m_dH
     + m_cDow[5] * m_dts
     + m_cDow[6] * m_dts * m_dts
     + m_cDow[7] * m_dts * m_LH
     + m_cDow[8] * m_dts * m_LH * m_LH
     + m_cDow[9] * m_deAlphaS
     + m_cDow[10] * m_deAlphaS * m_deAlphaS
     + m_cDow[11] * m_deAlphaS * m_LH
     + m_cDow[12] * m_deAlphaS * m_dts
     + m_cDow[13] * m_deAlpha
     + m_cDow[14] * m_deAlpha * m_LH
     + m_cDow[15] * m_dZ;
   
   return value / 1000.;
}

Double_t GEW::Z0Parametrisation::GaZstrangeParametrisation()
{
   Update();
   
   double value=0.0;
   
   value = m_cStr[0] 
     + m_cStr[1] * m_LH 
     + m_cStr[2] * m_LH * m_LH 
     + m_cStr[3] * m_dH 
     + m_cStr[4] * m_dH * m_dH
     + m_cStr[5] * m_dts
     + m_cStr[6] * m_dts * m_dts
     + m_cStr[7] * m_dts * m_LH
     + m_cStr[8] * m_dts * m_LH * m_LH
     + m_cStr[9] * m_deAlphaS
     + m_cStr[10] * m_deAlphaS * m_deAlphaS
     + m_cStr[11] * m_deAlphaS * m_LH
     + m_cStr[12] * m_deAlphaS * m_dts
     + m_cStr[13] * m_deAlpha
     + m_cStr[14] * m_deAlpha * m_LH
     + m_cStr[15] * m_dZ;
   
   return value / 1000.;
}

Double_t GEW::Z0Parametrisation::GaZcharmParametrisation()
{
   Update();
   
   double value=0.0;
   
   value = m_cCha[0] 
     + m_cCha[1] * m_LH 
     + m_cCha[2] * m_LH * m_LH 
     + m_cCha[3] * m_dH 
     + m_cCha[4] * m_dH * m_dH
     + m_cCha[5] * m_dts
     + m_cCha[6] * m_dts * m_dts
     + m_cCha[7] * m_dts * m_LH
     + m_cCha[8] * m_dts * m_LH * m_LH
     + m_cCha[9] * m_deAlphaS
     + m_cCha[10] * m_deAlphaS * m_deAlphaS
     + m_cCha[11] * m_deAlphaS * m_LH
     + m_cCha[12] * m_deAlphaS * m_dts
     + m_cCha[13] * m_deAlpha
     + m_cCha[14] * m_deAlpha * m_LH
     + m_cCha[15] * m_dZ;
   
   return value / 1000.;
}

Double_t GEW::Z0Parametrisation::GaZbottomParametrisation()
{
   Update();
   
   double value=0.0;
   
   value = m_cBot[0] 
     + m_cBot[1] * m_LH 
     + m_cBot[2] * m_LH * m_LH 
     + m_cBot[3] * m_dH 
     + m_cBot[4] * m_dH * m_dH
     + m_cBot[5] * m_dts
     + m_cBot[6] * m_dts * m_dts
     + m_cBot[7] * m_dts * m_LH
     + m_cBot[8] * m_dts * m_LH * m_LH
     + m_cBot[9] * m_deAlphaS
     + m_cBot[10] * m_deAlphaS * m_deAlphaS
     + m_cBot[11] * m_deAlphaS * m_LH
     + m_cBot[12] * m_deAlphaS * m_dts
     + m_cBot[13] * m_deAlpha
     + m_cBot[14] * m_deAlpha * m_LH
     + m_cBot[15] * m_dZ;
   
   return value / 1000.;
}


void GEW::Z0Parametrisation::Initialise()
{
   // table 4 from (i) 
   // non-factorizable EW*QCD corrections
   m_EWQCD[0] = 0;           // neutrino
   m_EWQCD[1] = 0;           // electron, muon, tau
   m_EWQCD[2] = -0.113e-3;   // up
   m_EWQCD[3] = -0.160e-3;   // down
   m_EWQCD[4] =  m_EWQCD[2]; // charm
   m_EWQCD[5] =  m_EWQCD[3]; // strange
   m_EWQCD[6] = -0.040e-3;   // bottom
 
   // table 4 from (i)
   // imaginary part of Z/gamma interfernce term
   m_ImKappa[0] = 0;         // neutrino
   m_ImKappa[1] = 0.0000368; // electron, muon, tau
   m_ImKappa[2] = 0.0000146; // up
   m_ImKappa[3] = 0.0000032; // down
   m_ImKappa[4] = 0.0000146; // charm
   m_ImKappa[5] = 0.0000032; // strange
   m_ImKappa[6] = 0.0000026; // bottom

   // From A.Freitas http://arxiv.org/abs/1401.2447 Table 7
   // Ele
   m_cEle[0] = 83.966 ;
   m_cEle[1] = -0.1017 ;
   m_cEle[2] = -0.06352 ;
   m_cEle[3] = 0.05500 ;
   m_cEle[4] = -0.00145 ;
   m_cEle[5] = 0.8051 ;
   m_cEle[6] = -0.027 ;
   m_cEle[7] = -0.017 ;
   m_cEle[8] = 0.0066 ;
   m_cEle[9] = -0.095 ;
   m_cEle[10] = -0.010 ;
   m_cEle[11] = -0.015 ;
   m_cEle[12] = 0.23 ;
   m_cEle[13] = -1.1 ;
   m_cEle[14] = 0.064 ;
   m_cEle[15] = 285 ;

   // Tau
   m_cTau[0] = 83.776 ;
   m_cTau[1] = -0.1016 ;
   m_cTau[2] = -0.06339 ;
   m_cTau[3] = 0.05488 ;
   m_cTau[4] = -0.00145 ;
   m_cTau[5] = 0.8036 ;
   m_cTau[6] = -0.026 ;
   m_cTau[7] = -0.017 ;
   m_cTau[8] = 0.0066 ;
   m_cTau[9] = -0.095 ;
   m_cTau[10] = -0.010 ;
   m_cTau[11] = -0.015 ;
   m_cTau[12] = 0.23 ;
   m_cTau[13] = -1.1 ;
   m_cTau[14] = 0.064 ;
   m_cTau[15] = 285 ;

   // Neu
   m_cNeu[0] = 167.157 ;
   m_cNeu[1] = -0.1567 ;
   m_cNeu[2] = -0.1194 ;
   m_cNeu[3] = 0.1031 ;
   m_cNeu[4] = -0.00269 ;
   m_cNeu[5] = 1.258 ;
   m_cNeu[6] = -0.13 ;
   m_cNeu[7] = -0.020 ;
   m_cNeu[8] = 0.0133 ;
   m_cNeu[9] = -0.19 ;
   m_cNeu[10] = -0.018 ;
   m_cNeu[11] = -0.021 ;
   m_cNeu[12] = 0.34 ;
   m_cNeu[13] = -0.084 ;
   m_cNeu[14] = 0.064 ;
   m_cNeu[15] = 503 ;

   // Up
   m_cUp[0] = 299.936 ;
   m_cUp[1] = -0.5681 ;
   m_cUp[2] = -0.2636 ;
   m_cUp[3] = 0.2334 ;
   m_cUp[4] = -0.00592 ;
   m_cUp[5] = 4.057 ;
   m_cUp[6] = -0.50 ;
   m_cUp[7] = -0.058 ;
   m_cUp[8] = 0.0352 ;
   m_cUp[9] = 14.26 ;
   m_cUp[10] = 1.6 ;
   m_cUp[11] = -0.081 ;
   m_cUp[12] = 1.7 ;
   m_cUp[13] = -11.1 ;
   m_cUp[14] = 0.19 ;
   m_cUp[15] = 1251 ;

   // Cha
   m_cCha[0] = 299.859 ;
   m_cCha[1] = -0.5680 ;
   m_cCha[2] = -0.2635 ;
   m_cCha[3] = 0.2334 ;
   m_cCha[4] = -0.00592 ;
   m_cCha[5] = 4.056 ;
   m_cCha[6] = -0.50 ;
   m_cCha[7] = -0.058 ;
   m_cCha[8] = 0.0352 ;
   m_cCha[9] = 14.26 ;
   m_cCha[10] = 1.6 ;
   m_cCha[11] = -0.081 ;
   m_cCha[12] = 1.7 ;
   m_cCha[13] = -11.1 ;
   m_cCha[14] = 0.19 ;
   m_cCha[15] = 1251 ;

   // Dow
   m_cDow[0] = 382.770 ;
   m_cDow[1] = -0.6199 ;
   m_cDow[2] = -0.3182 ;
   m_cDow[3] = 0.2800 ;
   m_cDow[4] = -0.00711 ;
   m_cDow[5] = 3.810 ;
   m_cDow[6] = -0.25 ;
   m_cDow[7] = -0.060 ;
   m_cDow[8] = 0.0420 ;
   m_cDow[9] = 10.20 ;
   m_cDow[10] = -2.4 ;
   m_cDow[11] = -0.083 ;
   m_cDow[12] = 0.65 ;
   m_cDow[13] = -10.1 ;
   m_cDow[14] = 0.19 ;
   m_cDow[15] = 1468 ;

   // Bot
   m_cBot[0] = 375.723 ;
   m_cBot[1] = -0.5744 ;
   m_cBot[2] = -0.3074 ;
   m_cBot[3] = 0.2725 ;
   m_cBot[4] = -0.00703 ;
   m_cBot[5] = -2.292 ;
   m_cBot[6] = -0.027 ;
   m_cBot[7] = -0.013 ;
   m_cBot[8] = 0.0428 ;
   m_cBot[9] = 10.53 ;
   m_cBot[10] = -2.4 ;
   m_cBot[11] = -0.088 ;
   m_cBot[12] = 1.2 ;
   m_cBot[13] = -10.1 ;
   m_cBot[14] = 0.19 ;
   m_cBot[15] = 1456 ;

   for (int i=0; i<16; ++i) {
     m_cMuo[i] = m_cEle[i];
     m_cStr[i] = m_cDow[i];
   }
}
