/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : Z0Parametrisation                                                     *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory of the parametrisation option                            *
 *      Computes effective weak mixing angle and Partial Z widths                 *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef GEW_Z0Parametrisation
#define GEW_Z0Parametrisation

#include "Gfitter/GConstants.h"

#include "GEW/Z0Base.h"

using namespace Gfitter;


namespace GEW {

   class Sin2ThetaF;
   class RadiatorFunctions;
   class DAlphaQED;
   class MH;
   
   class Z0Parametrisation : public Z0Base {

   public:
      
      Z0Parametrisation();
      virtual ~Z0Parametrisation() {}
      
      void Initialise();

      // accessors
      virtual Double_t GetSin2Eff(Gfitter::GTypes::Particle p);
      virtual Double_t GetGaZelectron();
      virtual Double_t GetGaZmuon();
      virtual Double_t GetGaZtau();
      virtual Double_t GetGaZneutrino();
      virtual Double_t GetGaZup();
      virtual Double_t GetGaZdown();
      virtual Double_t GetGaZstrange();
      virtual Double_t GetGaZcharm();
      virtual Double_t GetGaZbottom();
      virtual Double_t GetGaZhad();
      virtual Double_t GetDeltaS();
      virtual Double_t GetDeltaT();

      // effective vector and axial couplings
      virtual Double_t GetgAeff( Gfitter::GTypes::Particle particle, Double_t T3, Double_t ch );
      virtual Double_t GetgVeff( Gfitter::GTypes::Particle particle, Double_t T3, Double_t ch );

      // Use full 2-loop fermionic parametrization from A.Freitas
      virtual Bool_t DoFullTwoLoop() { return m_fullTwoLoop; }
      virtual void   SetFullTwoLoop( const Bool_t& fullTwoLoop ) { m_fullTwoLoop = fullTwoLoop; }

   protected:

      // full 2-loop fermionic parametrization from A.Freitas
      virtual Double_t GaZelectronParametrisation();
      virtual Double_t GaZmuonParametrisation();
      virtual Double_t GaZtauParametrisation();
      virtual Double_t GaZneutrinoParametrisation();
      virtual Double_t GaZupParametrisation();
      virtual Double_t GaZdownParametrisation();
      virtual Double_t GaZstrangeParametrisation();
      virtual Double_t GaZcharmParametrisation();
      virtual Double_t GaZbottomParametrisation();

      // partyial Z weidth
      Double_t GammaZff( Gfitter::GTypes::Particle particle, Double_t ch, Double_t mf =0  );  

      // update of flags in case of parameter change
      void UpdateLocalFlags( Gfitter::GReference& ref );
   
      // update parameters
      void Update();

      // parameters from the Data-Card
      Gfitter::GParameterRef p_mt;
      Gfitter::GParameterRef p_MZ;

      // parameters for theory uncertainties
      Gfitter::GParameterRef p_DeltaGammaZlept_Scale;
      Gfitter::GParameterRef p_DeltaGammaZnu_Scale;
      Gfitter::GParameterRef p_DeltaGammaZds_Scale;
      Gfitter::GParameterRef p_DeltaGammaZuc_Scale;
      Gfitter::GParameterRef p_DeltaGammaZb_Scale;
  
      // reference to DAlphaQED
      DAlphaQED& GetDAlphaQED() const { return (DAlphaQED&)*t_DAlphaQED; }      
      Gfitter::GTheoryRef    t_DAlphaQED; 
      
      // reference to AlphasMZ
      Gfitter::GTheory& GetAlphasMZ() const { return (Gfitter::GTheory&)*t_AlphasMZ; } 
      Gfitter::GTheoryRef    t_AlphasMZ; 
      
      // reference to Higgs mass
      GEW::MH& GetMH() const{ return (GEW::MH&)*t_MH; }
      Gfitter::GTheoryRef t_MH;

      // calculation settings
      Bool_t m_logMH;
      Bool_t m_fullTwoLoop;
  
   private:

      // axial and vector coupling
      Double_t gVf( GTypes::Particle ParticleType, Double_t T3 );
      Double_t gAf( GTypes::Particle ParticleType, Double_t T3 );
      
      // boolean flags to track recomputation
      Bool_t m_isUpToDate_Update; 

      Double_t m_Delta_SZ;
      Double_t m_Delta_TZ;

      // left/right couplings
      double m_gLv;
      double m_gRv;
      double m_gLe;
      double m_gRe;
      double m_gLu;
      double m_gRu;
      double m_gLd;
      double m_gRd;
      double m_gLb;
      double m_gRb;
   
      // non-factrorizable EW*QCD corrctions to Z->ff
      Double_t m_EWQCD[7];
        
      // imaginary part of Z/gamma interfernce term
      Double_t m_ImKappa[7];

      // axial and vector radiator functions
      Double_t m_CV[7];
      Double_t m_CA[7];

      // for parametrized 2-loop calculation
      Double_t m_LH;
      Double_t m_dH;
      Double_t m_dZ;
      Double_t m_dts;
      Double_t m_deAlpha;
      Double_t m_deAlphaS;

      Double_t m_cEle[16];
      Double_t m_cMuo[16];
      Double_t m_cTau[16];
      Double_t m_cNeu[16];
      Double_t m_cUp[16];
      Double_t m_cDow[16];
      Double_t m_cStr[16];
      Double_t m_cCha[16];
      Double_t m_cBot[16];

      // reference to radiator function
      RadiatorFunctions& GetRadFun() const { return (RadiatorFunctions&)*t_radFun; }
      Gfitter::GTheoryRef    t_radFun;
      
      ClassDef(Z0Parametrisation,0)
   };
}

#endif
