/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : LED_Delta_eHad                                                        *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * Sources:                                                                       *
 *      - R. Contino, L. Pilo, R. Rattazzi and A. Strumia                         *
 *        JHEP 0106, 005 (2001), [arXiv:hep-ph/0103104]                           *
 *      - [ADLO+SLD], Phys. Rept. 427, 257 (2006), [arXiv:hep-ex/0509008]         *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/
#include "Gfitter/GConstants.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"

#include "GSTU/LED_Delta_eHad.h"

#include "TMath.h"

using namespace Gfitter;

ClassImp(GSTU::LED_Delta_eHad)


   GSTU::LED_Delta_eHad::LED_Delta_eHad(): Gfitter::GTheory()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   Int_t value = gStore()->GetVariable( "GSTUFlags::LED_Ndim" )->GetIntValue();
   m_logger << kINFO << "Using: \"" << value << "\" dimensions for large extra dimensions!" << GEndl;
   m_ndim = value;

   bool value2 = gStore()->GetVariable( "GSTUFlags::LED_LogScale" )->GetBoolValue();
   m_logger << kINFO << "Using: \"" << value2 << "\" logarithmic scale!" << GEndl;
   m_logScale = value2;
   
   if( m_logScale ){
      BookParameter( "logMD",      &p_logMD );
      BookParameter( "logLambdaMD", &p_logLambdaMD );
   }
   else{
      BookParameter( "MD",       &p_MD );
      BookParameter( "LambdaMD", &p_LambdaMD );
   }

   BookTheory( "GEW::DAlphaQED", &t_DAlphaQED );
   BookParameter( "MZ",          &p_MZ );
   BookParameter( "GF",          &p_GF );

 }

// for references see hep-ph/0103104 page 25-26
Double_t GSTU::LED_Delta_eHad::GetTheoryPrediction()
{
   Double_t MD, lambdaMD;
   if( m_logScale ){
      MD       = std::pow( 10, p_logMD );
      lambdaMD = std::pow( 10, p_logLambdaMD );
   }   
   else{
      MD       = p_MD;
      lambdaMD = p_LambdaMD;
   }   

   Double_t alphaMZ = GConstants::alphaQED()/(1. - GetDAlphaQED().DAlphaQEDMZt() );
   
   Double_t MZ = 0.001*p_MZ;

   // eq. (7.13) from  hep-ex/0509008
   Double_t s02    = 0.5*( 1. - TMath::Sqrt(1.-4.*(TMath::Pi())*alphaMZ/(TMath::Sqrt(2.)*p_GF*p_MZ*p_MZ)) );

   Double_t delta_e = ( s02*MZ*MZ/(MD*MD)*std::pow(lambdaMD, m_ndim)*(5*(8.+5*m_ndim))/
                        (48.*TMath::Gamma(2+m_ndim/2.)*std::pow( TMath::Pi(), 2-m_ndim/2. )) );
   
   return delta_e;
}
