/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : LittlestHiggs                                                         *
 *                                                                                *
 * Description:                                                                   *
 *      implementation                                                            *
 *                                                                                *
 * Source:                                                                        *
 *      - J. Hubisz, P. Meade, A. Noble and M. Perelstein,                        *
 *        JHEP 0601, 135 (2006), [arXiv:hep-ph/0506042]                           *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GConstants.h"

#include "GSTU/LittlestHiggs.h"
#include "GSTU/LHmtplus.h"

using namespace Gfitter;

ClassImp(GSTU::LittlestHiggs)

GSTU::LittlestHiggs::LittlestHiggs()
   : TheoryBase()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   // book SM parameters
   BookParameter( "mt"            ,&p_mt       ); 
   BookParameter( "MZ"            ,&p_MZ       );
  
   
   // book LH parameters
   BookParameter( "f"          ,&p_f         ); 
   BookParameter( "slambda"    ,&p_slambda   ); 
   BookParameter( "deltac"     ,&p_deltac    );
   BookParameter( "Tfermions"  ,&p_Tfermions );
   BookParameter( "Ndoublets"  ,&p_Ndoublets );

   BookTheory   ( "GEW::MW"           ,&t_MW       );
   BookTheory   ( "GSTU::LHmtplus"    ,&t_mtplus   );
}


void GSTU::LittlestHiggs::Initialise(){}

// eq. (3.28) from hep-ph/0506042
Double_t GSTU::LittlestHiggs::GetS()
{
   Double_t R = TMath::Sqrt(1./(p_slambda*p_slambda) - 1);

   Double_t s = 1./(3*TMath::Pi())*TMath::Power( R*p_mt / Get_mtplus(), 2 )
      *( -5/2.0 + TMath::Log( TMath::Power( Get_mtplus()/p_mt, 2 ) ) );

   return s;
}

// eq. (3.28) from hep-ph/0506042
Double_t GSTU::LittlestHiggs::GetU()
{
   Double_t R = TMath::Sqrt(1./(p_slambda*p_slambda) - 1);

   Double_t u = 5./(6*TMath::Pi())*TMath::Power( R*p_mt/Get_mtplus(), 2 );
      
   return u;
}

// eqs. (3.28), (3.34), and (3.40) from hep-ph/0506042
Double_t GSTU::LittlestHiggs::GetT()
{
   Double_t MW  = GetMW().GetTheoryPrediction();
   Double_t pi  = TMath::Pi();
   Double_t cw2 = TMath::Power( MW/p_MZ, 2 );
   Double_t sw2 = 1. - cw2;
   Double_t vev = GConstants::vev();
   Double_t g   = 2*MW/vev;
   Double_t R   = TMath::Sqrt(1./(p_slambda*p_slambda) - 1);

   Double_t t = 3./(8*pi*sw2*cw2)*TMath::Power( R*p_mt*p_mt/(Get_mtplus()*p_MZ), 2 )
      *( TMath::Log( TMath::Power( Get_mtplus()/p_mt, 2 ) ) - 1. + 0.5* R*R ); 
  
   t += - 1./(4*pi*sw2)*TMath::Power( vev/p_f, 2 )*( p_deltac + 9/4.*TMath::Log(4*pi/g) );

   t += - p_Ndoublets*p_Tfermions;
   
   return t;
}

// eq. (3.42) from hep-ph/0506042
Double_t GSTU::LittlestHiggs::Get_deltagLb()
{
   Double_t MW  = GetMW().GetTheoryPrediction();
   Double_t pi  = TMath::Pi();
   Double_t cw  = MW/p_MZ;
   Double_t sw2 = 1. - cw*cw;
   Double_t vev = TMath::Sqrt(1/(2*GConstants::GF()));
   Double_t g   = 2*MW/vev;
   Double_t R   = TMath::Sqrt(1./(p_slambda*p_slambda) - 1);

   Double_t dglb = g*GConstants::alphaQED()*TMath::Power( p_mt, 4 )/( cw*8*pi*sw2*TMath::Power( MW*Get_mtplus(), 2 ) )
      * R*R*TMath::Log( TMath::Power( Get_mtplus()/p_mt, 2 ) );

   return dglb;
}
