/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : RandallSundrum                                                        *
 *                                                                                *
 * Description:                                                                   *
 *      implementation                                                            *
 *                                                                                *
 * Sources:                                                                       *
 *      S. Casagrande, F. Goertz, U. Haisch, M. Neubert, T. Pfoh                  * 
 *            Flavor Physics in the Randall-Sundrum Model:                        *
 *            I. Theoretical Setup and Electroweak Precision Tests                *
 *            JHEP 0810, 094 (2008)                                               *
 *            arXiv:0807.4937                                                     *
 *                                                                                *
 *      A. Delgado and A. Falkowski                                               *
 *            JHEP 05, 097 (2007)                                                 *
 *            [hep-ph/0702234]                                                    *
 *                                                                                *
 *      M. S. Carena, A. Delgado, E. Ponton, T. M. P. Tait and C. E. M. Wagner    *
 *            Phys. Rev. D68, 035010 (2003)                                       *
 *            [hep-ph/0305188]                                                    *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GConstants.h"

#include "GSTU/RandallSundrum.h"

using namespace Gfitter;

ClassImp(GSTU::RandallSundrum)

GSTU::RandallSundrum::RandallSundrum()
   : TheoryBase()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   // book SM parameters
   BookTheory   ( "GEW::MW"   ,&t_MW );
   BookParameter( "MZ"        ,&p_MZ  );
   
   // book RS parameters
   BookParameter( "MKK" ,&p_MKK ); 
   BookParameter( "L"   ,&p_L   ); 
     
}


void GSTU::RandallSundrum::Initialise(){}

// equations refer to JHEP 0810, 094 (2008)
// eq. (147)
Double_t GSTU::RandallSundrum::GetS()
{
   Double_t vev = GConstants::vev();
   Double_t pi  = TMath::Pi();
   
   Double_t s   = 2.*pi*vev*vev/(p_MKK*p_MKK)*(1. - 1./p_L);

   return s;
}

// eq. (147)
Double_t GSTU::RandallSundrum::GetT()
{
   Double_t vev = GConstants::vev();
   Double_t pi  = TMath::Pi();
   Double_t MW  = GetMW().GetTheoryPrediction();
   Double_t cw2 = TMath::Power( MW/p_MZ, 2 );
   Double_t t   = pi*vev*vev/(2.*cw2*p_MKK*p_MKK)*(p_L - 1./(2.*p_L));

   return t;
}
