/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : TwoHDM                                                                *
 *                                                                                *
 * Description:                                                                   *
 *      implementation                                                            *
 *                                                                                *
 * Sources:                                                                       *
 *      H. E. Haber                                                               *
 *            hep-ph/9306207                                                      *
 *                                                                                *
 *      H. E. Haber and H. E. Logan                                               *
 *            Phys. Rev. D62, 015011 (2000)                                       *
 *            hep-ph/9909335                                                      *
 *                                                                                *
 *      C. D. Froggatt, R. G. Moorhouse and I. G. Knowles                         *
 *            Phys. Rev. D45, 2471 (1992)                                         *
 *                                                                                *
 *      H.-J. He, N. Polonsky, S. Su                                              *
 *            Extra Families, Higgs Spectrum and Oblique Corrections              *
 *            Phys Rev D 64, 053004, 2001                                         *
 *   	        arXiv:hep-ph/0102144v2                                              *
 *                                                                                * 
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GConstants.h"

#include "GSTU/TwoHDM.h"
#include "GSTU/LHmtplus.h"

using namespace Gfitter;

ClassImp(GSTU::TwoHDM)

GSTU::TwoHDM::TwoHDM()
   : TheoryBase()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   // book SM parameters
   BookParameter( "mt"            ,&p_mt       ); 
   BookParameter( "MZ"            ,&p_MZ       );
  
	// book LH parameters
   BookParameter( "MH"            ,&p_MH       );
   BookParameter( "MH0"           ,&p_MH0      );
   BookParameter( "MA"            ,&p_MA       );
   BookParameter( "MHcharge"      ,&p_MHcharge );
	BookParameter( "betaminusalpha"    ,&p_betaminusalpha );

   BookTheory   ( "GEW::MW"           ,&t_MW       );
}


void GSTU::TwoHDM::Initialise(){}

// equations refer to Phys Rev D 64, 053004, 2001
// eq. (A.1)
Double_t GSTU::TwoHDM::GetS()
{
  Double_t MZ2 = p_MZ*p_MZ;
  Double_t MH2 = p_MH*p_MH;
  Double_t MH02 = p_MH0*p_MH0;
  Double_t MHc2 = p_MHcharge*p_MHcharge;
  Double_t MA2 = p_MA*p_MA;

  Double_t s = 1./(TMath::Pi()*MZ2)*( pow( sin(p_betaminusalpha), 2 )*B22(MZ2,MH02,MA2)
												 - B22(MZ2,MHc2,MHc2)
												 + pow( cos(p_betaminusalpha), 2 )*( B22(MZ2,MH2,MA2)
																								 + B22(MZ2,MZ2,MH02)
																								 - B22(MZ2,MZ2,MH2)
																								 - MZ2*B0(MZ2,MZ2,MH02)
																								 + MZ2*B0(MZ2,MZ2,MH2)));
  if( p_MH == 400 )
	 std::cout << p_MH << "&\t" << p_MH0 << "&\t" << p_MHcharge << "&\t" << p_MA << "&\t" << p_betaminusalpha << "&\t"
				  << GetS() << "&\t"					
				  << GetT() << "&\t"
				  << GetU() << "\\\\ \n";

  return s;
}

// eq. (A.2)
Double_t GSTU::TwoHDM::GetT()
{
  Double_t MW2 = pow( GetMW().GetTheoryPrediction(), 2);
  Double_t MZ2 = p_MZ*p_MZ;
  Double_t sw2 = 1 - MW2/MZ2;

  Double_t MH2 = p_MH*p_MH;
  Double_t MH02 = p_MH0*p_MH0;
  Double_t MHc2 = p_MHcharge*p_MHcharge;
  Double_t MA2 = p_MA*p_MA;

  Double_t t = 1./(16.*TMath::Pi()*MW2*sw2)*(F(MHc2,MA2)
															+ pow( sin(p_betaminusalpha), 2)*(F(MHc2,MH02) - F(MA2,MH02))
															+ pow( cos(p_betaminusalpha), 2)*(F(MHc2,MH2) - F(MA2,MH2)
																										 + F(MW2,MH02) - F(MW2,MH2)
																										 - F(MZ2,MH02) + F(MZ2,MH2)
																										 + 4*MZ2*Bbar0(MZ2,MH02,MH2)
																										 - 4*MW2*Bbar0(MW2,MH02,MH2)));

  return t;
}

// eq. (A.3)
Double_t GSTU::TwoHDM::GetU()
{
  Double_t MW2 = pow( GetMW().GetTheoryPrediction(), 2);
  Double_t MZ2 = p_MZ*p_MZ;
  Double_t MH2 = p_MH*p_MH;
  Double_t MH02 = p_MH0*p_MH0;
  Double_t MHc2 = p_MHcharge*p_MHcharge;
  Double_t MA2 = p_MA*p_MA;

  Double_t u = - GetS()
	 + 1./(TMath::Pi()*MZ2)*(B22(MW2,MA2,MHc2) - 2*B22(MW2,MHc2,MHc2)
									 + pow( sin(p_betaminusalpha), 2)*B22(MW2,MH02,MHc2)
									 + pow( cos(p_betaminusalpha), 2)*(B22(MW2,MH2,MHc2)
																				  + B22(MW2,MW2,MH02)
																				  - B22(MW2,MW2,MH2)
																				  - MW2*B0(MW2,MW2,MH02)
																				  + MW2*B0(MW2,MW2,MH2)));
  return u;
}

// eq. (A.7) 
Double_t GSTU::TwoHDM::B22( Double_t q2, Double_t m12, Double_t m22)
{
  Double_t x1 = m12/q2;
  Double_t x2 = m22/q2;
  
  Double_t b22 = 0;
  
  if( x1 == x2 ) 
	 b22 = q2/24.*(2*TMath::Log(q2) + 2*TMath::Log(x1) + (16*x1 - 10/3) + (4*x1 - 1)*G(x1));
  else
	 b22 = q2/24.*( 2*TMath::Log(q2) + TMath::Log(x1*x2)
						 + ( pow(( x1 - x2 ), 3 ) - 3*( x1*x1 - x2*x2 ) + 3*( x1 - x2 ))*TMath::Log(x1/x2)
						 - ( 2*pow(( x1 - x2 ), 2) - 8*( x1 + x2 ) + 10/3 )
						 - ( pow(( x1 - x2 ), 2) - 2*( x1 + x2 ) + 1 )*f(x1,x2)
						 - 6*F(x1,x2));
  
  if( isnan(b22) )
	 std::cout << "b22 is NaN" << std::endl;
  
  return b22;
}

// eq. (A.4)
Double_t GSTU::TwoHDM::B0( Double_t q2, Double_t m12, Double_t m22)
{
  Double_t x1 = m12/q2;
  Double_t x2 = m22/q2;

  Double_t b0 = 0;

  if( m12 == m22 )
	 b0 = 2.-2.*sqrt(4*x1-1.)*atan(1./(sqrt(4*x1-1.)));
  else
	 b0 = 1 + 1/2*( (x1 + x2 )/( x1 - x2 ) - ( x1 - x2 ))*TMath::Log(x1/x2) + 1/2*f(x1,x2); 

  if( isnan(b0) )
	 std::cout << "b0 is NaN" << std::endl;

  return b0;
}

// eq. (A.5)
Double_t GSTU::TwoHDM::Bbar0( Double_t m12, Double_t m22, Double_t m32)
{
  Double_t bbar01 = 0;
  Double_t bbar02 = 0;

  //Limits for mi2 == mj2 - own calculation, not taken from Phys Rev D 64, 053004

  if( m12 == m32 )
	 bbar01 = TMath::Log(m12) + 1;
  else 
	 bbar01 = ( m12*TMath::Log(m12)-m32*TMath::Log(m32) )/( m12 - m32 );

  if( m12 == m22 )
	 bbar02 = TMath::Log(m12) + 1;
  else
	 bbar02 = ( m12*TMath::Log(m12)-m22*TMath::Log(m22) )/( m12 - m22 );

  if( isnan(bbar01) || isnan(bbar02) )
	 std::cout << "bbar0 is NaN" << std::endl;

  return bbar01 - bbar02;
}

// eq. (A.10)
Double_t GSTU::TwoHDM::G( Double_t x)
{
  return -4.*sqrt( 4.*x - 1. )*atan( 1./sqrt( 4.*x - 1. ));
}

// eq. (A.9)
Double_t GSTU::TwoHDM::F( Double_t x1, Double_t x2)
{
  return 1./2.*( x1 + x2 ) - x1*x2*TMath::Log( x1/x2 )/( x1 - x2 );
}

// eq. (A.11)
Double_t GSTU::TwoHDM::f( Double_t x1, Double_t x2)
{
  Double_t delta = Delta( x1, x2);

  if ( delta == 0 )
	 return 0;
  
  else if ( delta > 0 )
	 return -2.*sqrt(delta)*( atan(( x1 - x2 + 1. )/sqrt(delta)) - atan(( x1 - x2 - 1. )/sqrt(delta)));
  
  else
	 return sqrt(-delta)*TMath::Log(( x1 + x2 - 1. + sqrt(-delta))/(x1 + x2 - 1. - sqrt(-delta)));

}

// eq. (A.12) 
Double_t GSTU::TwoHDM::Delta( Double_t x1, Double_t x2)
{
  return 2.*( x1 + x2 ) - ( x1 - x2 )*( x1 - x2 ) - 1.;
}



