/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GMath                                                                 *
 *                                                                                *
 * Description:                                                                   *
 *      Basic math operations missing in TMath                                    *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef Gfitter_GMath
#define Gfitter_GMath

#include <assert.h>
#include <complex>

#include "Rtypes.h"
#include "TMatrixD.h"

namespace Gfitter {

   namespace GMath {

      // integer power
      Double_t IPow( const Double_t& val, const Int_t& exp );

      // sqrt(2) - in TMath, but apparently not in older ROOT versions ?
      Double_t Sqrt2();

      // Riemann zeta function
      Double_t Zeta2();
      Double_t Zeta3();
      Double_t Zeta4();
      Double_t Zeta5();

      // 1/3, 2/3
      Double_t OneThird();
      Double_t TwoThird();

      // ---- matrix and covariance operations
      // turns covariance into correlation matrix
      TMatrixD* GetCorrelationMatrix( const TMatrixD& covMat );

      // Asymetric Gauss 
      Double_t AsymGauss(Double_t *x, Double_t *params);

      // NONE value
      Double_t None();
   }
}

// ------------- inlined functions for fast access -----------------------------

// integer power
inline Double_t Gfitter::GMath::IPow( const Double_t& val, const Int_t& exp ) 
{
   return std::pow( val, exp );
}      
   
// sqrt(2) - in TMath, but apparently not in older ROOT versions ?
inline Double_t Gfitter::GMath::Sqrt2() 
{ 
   return 1.4142135623730950488016887242097; 
}

// Riemann zeta Function
inline Double_t Gfitter::GMath::Zeta2()
{
   return 1.644934066848226; // pi*pi/6
}

inline Double_t Gfitter::GMath::Zeta3()
{
   return 1.202056903159594;
}

inline Double_t Gfitter::GMath::Zeta4()
{
   return 1.082323233711138; // pi^4/90
}

inline Double_t Gfitter::GMath::Zeta5()
{
   return 1.036927755143370;
}

// 1/3
inline Double_t Gfitter::GMath::OneThird()
{
   return 0.3333333333333333;
}

// 2/3
inline Double_t Gfitter::GMath::TwoThird()
{
   return 0.6666666666666666;
}

inline Double_t Gfitter::GMath::None() 
{ 
   return -1234.567; 
}

#endif      
