/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GMultiNestFitter                                                         *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include "Riostream.h"
#include "Gfitter/GMultiNestFitter.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GData.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GResultEval.h"
#include <math.h>
#include "TMath.h"

using namespace std;

ClassImp(Gfitter::GMultiNestFitter)

Gfitter::GMultiNestFitter::GMultiNestFitter() 
  : GFitterBase(),
    m_isFirstInit( kTRUE )
{
  //default parameters
  m_nlive = 10000;	
  m_efr = 0.8;		
  m_tol = 0.1;		
  m_ceff = false;
  m_maxiter = 1000;
  m_fb=false;
  m_resume=false;
  m_outfile=false;
  m_outfilename="MyMultiNestTest";

  InitClassName( "GMultiNestFitter" );
}

Gfitter::GMultiNestFitter::~GMultiNestFitter()
{

}

void Gfitter::GMultiNestFitter::Initialise()
{

   // run the prefitter first
   // if (gStore()->GetPreFitter()) {
   //    gStore()->GetPreFitter()->Initialise();
   //    gStore()->GetPreFitter()->ExecuteFit();
   // }

   m_This = this;      

   // base class call
   GFitterBase::Initialise();
   
   if( !m_isFirstInit )
      GetThisPtr()->SetFirstPassed();
   
   // reset vectors
   GetFitPars().clear();
   GetPenalisedPars().clear();      
   

   //count fit parameters
   Int_t npars = 0;
   GParPtrVec_t::const_iterator par = gStore()->GetActiveParameters().begin();
   for (; par != gStore()->GetActiveParameters().end(); ++par) {
     if ((*par)->VariesInFit()) {
       GetFitPars().push_back( *par );
       npars++;
     }
   }
   // set internally the number of fit parameters (needed due to TFitter "feature" of 
   // "current" free parameters given to FCN only)
   this->SetNumOfFitPars( npars );


  // JH: loop over all Parameters needed here! Scan!
   for (par = gStore()->GetParameters().begin(); par != gStore()->GetParameters().end(); ++par) {
      // change from MG: We do not need IsFirst(), don't we?
      if ((*par)->HasTheory() && (*par)->IsScanned() ) { 
         m_logger << kINFO << "Setting penalised Parameter: " << (*par)->GetParName() << GEndl;
         GetPenalisedPars().push_back( (*par) );
      }
   }  

   //read in parameters from input file
   if (gStore()->ExistVariable( "MultinestFitter::NLivePoints" )) {
      m_nlive = gStore()->GetVariable( "MultinestFitter::NLivePoints" )->GetIntValue();
   }
   if (gStore()->ExistVariable( "MultinestFitter::Tolerance" )) {
      m_tol = gStore()->GetVariable( "MultinestFitter::Tolerance" )->GetFloatValue();
   }
   if (gStore()->ExistVariable( "MultinestFitter::ConstantEfficiency" )) {
      m_ceff = gStore()->GetVariable( "MultinestFitter::ConstantEfficiency" )->GetBoolValue();
   }
   if (gStore()->ExistVariable( "MultinestFitter::Efficiency" )) {
      m_efr = gStore()->GetVariable( "MultinestFitter::Efficiency" )->GetFloatValue();
   }
   if (gStore()->ExistVariable( "MultinestFitter::MaxIterations" )) {
      m_maxiter = gStore()->GetVariable( "MultinestFitter::MaxIterations" )->GetIntValue();
      if(m_maxiter<1000) m_maxiter=1000; //use at least 1000 iterations, otherwise timer status will crash
   }
   if (gStore()->ExistVariable( "MultinestFitter::PrintFeedback" )) {
      m_fb = gStore()->GetVariable( "MultinestFitter::PrintFeedback" )->GetBoolValue();
   }
   if (gStore()->ExistVariable( "MultinestFitter::Outfile" )) {
      m_outfile = gStore()->GetVariable( "MultinestFitter::Outfile" )->GetBoolValue();
   }
   if (gStore()->ExistVariable( "MultinestFitter::Resume" )) {
     m_resume = gStore()->GetVariable( "MultinestFitter::Resume" )->GetBoolValue();
   }
   if( gStore()->ExistVariable( "MultinestFitter::OutfileName" )) {
     m_outfilename = gStore()->GetVariable( "MultinestFitter::OutfileName" )->GetStringValue();
   }
 
   // some debug output, only when function is called first
   if( m_isFirstInit ){
      PrintConfiguration();
      m_isFirstInit = kFALSE;
   }

   //reset counter of iterations
   m_iter=0;

   // init covariance matrices and/or diagonal errors
   if (gStore()->IsCorrelationActive()) this->InitErrorMatrix();

}
   
Double_t Gfitter::GMultiNestFitter::GetCurrentChi2()
{ 
  return m_chi2;
}

Double_t Gfitter::GMultiNestFitter::ExecuteFit()
{   

  // set the MultiNest sampling parameters

  int IS = 1;					// do Nested Importance Sampling?
  
  int mmodal = 0;					// do mode separation?
  
  int ndims =  GetNumOfFitPars();				// dimensionality (no. of free parameters)
  
  int nPar =  gStore()->GetActiveParameters().size();					// total no. of parameters including free & derived parameters
  
  int nClsPar = GetNumOfFitPars();				// no. of parameters to do mode separation on
  
  m_updInt = m_maxiter/1000;				// after how many iterations feedback is required & the output files should be updated
  if(m_updInt>10) m_updInt=10;

  if(m_outfile) m_updInt=m_maxiter;                   //in case of outfile -> only call dumper routine once at the end to avoid lare I/O, but timer called in dumper routine will not work

  // note: posterior files are updated & dumper routine is called after every updInt*10 iterations
  
  double Ztol = -1E90;				// all the modes with logZ < Ztol are ignored
  
  int maxModes = 100;				// expected max no. of modes (used only for memory allocation)
  
  int pWrap[ndims];				// which parameters to have periodic boundary conditions?
  for(int i = 0; i < ndims; i++) pWrap[i] = 0;
  
  const char* root = m_outfilename;         	 		// root for output files
  
  int seed = -1;					// random no. generator seed, if < 0 then take the seed from system clock
  
  bool fb = m_fb;					// need feedback on standard output?
  
  bool resume = m_resume;				// resume from a previous job?
  
  bool outfile = m_outfile;				// write output files?
  
  int initMPI = 1;				// initialize MPI routines?, relevant only if compiling with MPI
  // set it to F if you want your main program to handle MPI initialization
  
  double logZero = -1E90;				// points with loglike < logZero will be ignored by MultiNest
  
  
  void *context = 0;				// not required by MultiNest, any additional information user wants to pass
  
  // init timer
  m_timer = new GTimer( m_maxiter/(m_updInt*10), "MultiNestTimer");
  // calling MultiNest
  nested::run(IS, mmodal, m_ceff, m_nlive, m_tol, m_efr, ndims, nPar, nClsPar, maxModes, m_updInt, Ztol, root, seed, pWrap, fb, resume, outfile, initMPI,
  	      logZero, m_maxiter, LogLike, dumper, context);

  //print out the result
  std::vector<Double_t> gvec;
  std::vector<TString>  svec;
  for (UInt_t ipar=0; ipar<GetFitPars().size(); ipar++) {
    gvec.push_back( GetFitPars()[ipar]->GetFitValue() );
    svec.push_back( GetFitPars()[ipar]->GetParName() );
  }
  m_logger <<  kINFO ;
  GUtils::FormattedOutput( gvec, svec, "Parameters", "Best values", m_logger, "%+1.4g" );
  m_logger << GEndl;
  m_logger << kINFO << "Elapsed time: " << GetFitDuration() << GEndl;
 
  Double_t chi2=GetCurrentChi2();
  return chi2;
}

void Gfitter::GMultiNestFitter::DrawContour( GParameter* gpar1, GParameter* gpar2, 
                                          Int_t npoints, Double_t dchi2, Double_t* npx, Double_t* npy )
{

}

void Gfitter::GMultiNestFitter::UpdateResults()
{
  //content of this function has been moved to dumper routine which is automatically called by multinest at the end of the fit
}


Gfitter::GMultiNestFitter* Gfitter::GMultiNestFitter::m_This = 0;

Gfitter::GMultiNestFitter* Gfitter::GMultiNestFitter::GetThisPtr()
{
   return m_This;
}

// ------------------------------------------------------------------------------------------
// MultiNest   W r a p p e r s   
// ------------------------------------------------------------------------------------------


void Gfitter::GMultiNestFitter::LogLike(double *Cube, int &ndim, int  &npars, double &lnew, void *context)
{
  // Input arguments
  // ndim 						= dimensionality (total number of free parameters) of the problem
  // npars 						= total number of free plus derived parameters
  // context						void pointer, any additional information
  //
  // Input/Output arguments
  // Cube[npars] 						= one entry has the ndim parameters in unit-hypercube
  //	 						on exit, the physical parameters plus copy any derived parameters you want to store with the free parameters
  //	 
  // Output arguments
  // lnew 						= loglikelihood


  //scale parameters from unit-hypercube to fit range of parameters
	
  Double_t fitPars[ndim];
  GParPtrVec_t::const_iterator par = gStore()->GetActiveParameters().begin();
  Int_t np = 0;
  Int_t nd = 0;
  for (; par != gStore()->GetActiveParameters().end(); ++par) {

    if ((*par)->VariesInFit()) {
      //std::cout << "Free parameter: " << (*par)->GetFullName() << std::endl;
      
      //use linear scaling in case of flat prior, otherwise use Gauss scaling
      if((*par)->GetErrGaussm()==0 && (*par)->GetErrGaussp()==0){
	Double_t max =  (*par)->GetScanRange().GetMax();
	Double_t min =  (*par)->GetScanRange().GetMin(); 
	fitPars[nd] = ScaleUniformPrior(Cube[nd],min,max);
      }
      else{
	fitPars[nd] = ScaleGaussPrior( Cube[nd], (*par)->GetValue(), (*par)->GetErrGaussp(), (*par)->GetErrGaussm()) ;
      }
      
      (*par)->SetFitValue( fitPars[nd]);
      nd++;
    }
    //np++;

  }

  Double_t Chi2;
  (GetThisPtr())->FCN( Chi2, fitPars, 0 );
  lnew = -0.5*Chi2;


  //store derived parameters after function evaluation (not really needed?)
  
  par = gStore()->GetActiveParameters().begin();
  np=0;
  for (; par != gStore()->GetActiveParameters().end(); ++par) {
    if (!(*par)->VariesInFit()) {
      Double_t max =  (*par)->GetScanRange().GetMax();
      Double_t min =  (*par)->GetScanRange().GetMin(); 

      double value;
      if (!(*par)->HasTheory()) value = (*par)->GetFitValue();
      else                      value = (*par)->GetTheoryPrediction();
      //std::cout << (*par)->GetFullName()<< "  " << value << "  "  << max << "  " << min <<  std::endl; 
      Cube[np+nd] = (value-min)/(max-min);
      np++;
    }
  }
  
	
}


/************************************************* dumper routine ******************************************************/

// The dumper routine will be called every updInt*10 iterations
// MultiNest doesn not need to the user to do anything. User can use the arguments in whichever way he/she wants
//
//
// Arguments:
//
// nSamples 						= total number of samples in posterior distribution
// nlive 						= total number of live points
// nPar 						= total number of parameters (free + derived)
// physLive[1][nlive * (nPar + 1)] 			= 2D array containing the last set of live points (physical parameters plus derived parameters) along with their loglikelihood values
// posterior[1][nSamples * (nPar + 2)] 			= posterior distribution containing nSamples points. Each sample has nPar parameters (physical + derived) along with their loglike value & posterior probability
// paramConstr[1][4*nPar]:
// paramConstr[0][0] to paramConstr[0][nPar - 1] 	= mean values of the parameters
// paramConstr[0][nPar] to paramConstr[0][2*nPar - 1] 	= standard deviation of the parameters
// paramConstr[0][nPar*2] to paramConstr[0][3*nPar - 1] = best-fit (maxlike) parameters
// paramConstr[0][nPar*3] to paramConstr[0][4*nPar - 1] = MAP (maximum-a-posteriori) parameters
// maxLogLike						= maximum loglikelihood value
// logZ							= log evidence value from the default (non-INS) mode
// INSlogZ						= log evidence value from the INS mode
// logZerr						= error on log evidence value
// context						void pointer, any additional information

void Gfitter::GMultiNestFitter::dumper(int &nSamples, int &nlive, int &nPar, double **physLive, double **posterior, double **paramConstr, double &maxLogLike, double &logZ, double &INSlogZ, double &logZerr, void *context)
{
  // measure time for fit, not called when outfile is defined
   if(! (GetThisPtr())->m_outfile){
     (GetThisPtr())->m_iter++;

    (GetThisPtr())->m_fitDuration = (GetThisPtr())->m_timer->GetElapsedTime();
    //(GetThisPtr())->m_logger << kINFO << " time left: " << (GetThisPtr())->m_timer->GetLeftTime( (GetThisPtr())->m_iter ) << GEndl;
    (GetThisPtr())->m_timer->DrawProgressBar( (GetThisPtr())->m_iter );

    //store scan result only in last iteration step but at least every 1000 steps
    if(  (GetThisPtr())->m_iter <  (GetThisPtr())->m_maxiter/((GetThisPtr())->m_updInt*10) &&  nSamples%1000!=0 ) return;
  }

  GParPtrVec_t::const_iterator par = gStore()->GetActiveParameters().begin();
  int i=0;
  int j=0;
  int k=0;
  int nf=0;

  // ---- store scan result ----

  (GetThisPtr())->m_postdist.clear();
  (GetThisPtr())->m_LivePts.clear();

  //scale parameters back from unit cube and store them
  //note: multinest re-orders the parameters, first free parameters then derived in the posterior list
  for (; par != gStore()->GetActiveParameters().end(); ++par) {
    if ((*par)->VariesInFit()) {
      nf++;
    }
  }
  par = gStore()->GetActiveParameters().begin();

  for (; par != gStore()->GetActiveParameters().end(); ++par) {
    if ((*par)->VariesInFit()) {
      if((*par)->GetErrGaussm()==0 && (*par)->GetErrGaussp()==0){
	Double_t max =  (*par)->GetScanRange().GetMax();
	Double_t min =  (*par)->GetScanRange().GetMin(); 
	for( j = 0; j < nSamples; j++ ){
	  posterior[0][i * nSamples + j] = ScaleUniformPrior(posterior[0][i * nSamples + j],min,max);
	  (GetThisPtr())->m_postdist.push_back( posterior[0][i * nSamples + j]);
	}
	for( j = 0; j < nlive; j++ ){
	  physLive[0][i * nlive + j] =  ScaleUniformPrior(physLive[0][i * nlive + j],min,max);
	  (GetThisPtr())->m_LivePts.push_back( physLive[0][i * nlive + j]);
	}
      }
      else{
	for( j = 0; j < nSamples; j++ ){
	  posterior[0][i * nSamples + j] = ScaleGaussPrior(posterior[0][i * nSamples + j] , (*par)->GetValue(), (*par)->GetErrGaussp(), (*par)->GetErrGaussm()) ;
	  (GetThisPtr())->m_postdist.push_back( posterior[0][i * nSamples + j]);
	}
	for( j = 0; j < nlive; j++ ){
	  physLive[0][i * nlive + j] = ScaleGaussPrior(physLive[0][i * nlive + j] , (*par)->GetValue(), (*par)->GetErrGaussp(), (*par)->GetErrGaussm()) ;
	  (GetThisPtr())->m_LivePts.push_back( physLive[0][i * nlive + j]);
	}
      }
      i++;
    }
    else{
      if((*par)->GetErrGaussm()==0 && (*par)->GetErrGaussp()==0){
	Double_t max =  (*par)->GetScanRange().GetMax();
	Double_t min =  (*par)->GetScanRange().GetMin(); 
	for( j = 0; j < nSamples; j++ ){
	  posterior[0][(nf+k) * nSamples + j] = ScaleUniformPrior(posterior[0][(nf+k) * nSamples + j],min,max);
	  (GetThisPtr())->m_postdist.push_back( posterior[0][(nf+k) * nSamples + j]);
	}
	for( j = 0; j < nlive; j++ ){
	  physLive[0][(nf+k) * nlive + j] =  ScaleUniformPrior(physLive[0][(nf+k) * nlive + j],min,max);
	  (GetThisPtr())->m_LivePts.push_back( physLive[0][(nf+k) * nlive + j]);
	}
      }
      else{
	for( j = 0; j < nSamples; j++ ){
	  posterior[0][(nf+k) * nSamples + j] = ScaleGaussPrior(posterior[0][(nf+k) * nSamples + j] , (*par)->GetValue(), (*par)->GetErrGaussp(), (*par)->GetErrGaussm()) ;
	  (GetThisPtr())->m_postdist.push_back( posterior[0][(nf+k) * nSamples + j]);
	}
	for( j = 0; j < nlive; j++ ){
	  physLive[0][(nf+k) * nlive + j] = ScaleGaussPrior(physLive[0][(nf+k) * nlive + j] , (*par)->GetValue(), (*par)->GetErrGaussp(), (*par)->GetErrGaussm()) ;
	  (GetThisPtr())->m_LivePts.push_back( physLive[0][(nf+k) * nlive + j]);
	}
      }
      k++;
    }
  }

  //store the values of loglike and posterior
  for( i = nPar; i < nPar + 2; i++ ){
     for( j = 0; j < nSamples; j++ ){
       (GetThisPtr())->m_postdist.push_back( posterior[0][i * nSamples + j]);
     }
  }
  for( i = nPar; i < nPar + 1; i++ ){   
     for( j = 0; j < nlive; j++ ){
       (GetThisPtr())->m_LivePts.push_back( physLive[0][i * nlive + j]);
     }
  }
  
  (GetThisPtr())->m_nPar = nPar;
  (GetThisPtr())->m_nSamples = nSamples;

  //std::cout << "dumper:       nPar = " << nPar <<std::endl;
  //std::cout << "dumper:       nSamples = " << nSamples <<std::endl;

  // ---- store fit result ----

  // the parameter vector
  const GParPtrVec_t& fp = (GetThisPtr())->GetFitPars();
  const UInt_t npar = fp.size();
  
  // retrieve covariance matrix first, not implement for multinest -> fill unit matrix
  TMatrixD* covMat = new TMatrixD( npar, npar );
  for (UInt_t i=0; i < npar; i++) {
    for (UInt_t j=0; j < npar; j++) {
      if(i==j) (*covMat)(i,j) = 1;
      else  (*covMat)(i,j) = 0;
    }
  }

  Int_t np = 0;
  nf = 0;
  
  par = gStore()->GetActiveParameters().begin();

  Double_t chi2 = -0.5*maxLogLike; 
  (GetThisPtr())->SetCurrentChi2(chi2);

  for (; par != gStore()->GetActiveParameters().end(); ++par) {
    if ((*par)->VariesInFit()) {

      Double_t val, errp, errm, errsym, globcor;

      //scale parameters back from unit cube
      if((*par)->GetErrGaussm()==0 && (*par)->GetErrGaussp()==0){
	Double_t max =  (*par)->GetScanRange().GetMax();
	Double_t min =  (*par)->GetScanRange().GetMin(); 
	val = ScaleUniformPrior(paramConstr[0][2*nPar+nf],min,max);
      }
      else{
	val = ScaleGaussPrior( paramConstr[0][2*nPar+nf], (*par)->GetValue(), (*par)->GetErrGaussp(), (*par)->GetErrGaussm()) ;
      }

      //std::cout << (*par)->GetFullName() << "   value = " << val << std::endl;
      errp = (*par)->GetErrGaussp();
      errm = (*par)->GetErrGaussm(); 
      errsym = paramConstr[0][nPar+nf];//seems to be pure RMS, scaling?

      globcor = 0; //update

      fp[nf]->SetResult( new GResultEval( val, chi2, errp, errm, errsym, globcor, covMat, (GetThisPtr())->GetFitDuration() ) );
      fp[nf]->SetFitValue(val);
      nf++;
    }
    np++;
  }


}

// Uniform[0:1]  ->  Uniform[x1:x2]
double Gfitter::GMultiNestFitter::ScaleUniformPrior(double r, double x1,double x2){
  return x1+r*(x2-x1);
}

// Uniform[0:1]  ->  Gaussian
double Gfitter::GMultiNestFitter::ScaleGaussPrior(double r, double mu, double sigmaplus, double sigmaminus){
  double sigma = r>mu ? sigmaplus : sigmaminus;
  double SqrtTwo = 1.414213562;
  if(r<=0.00001) r=0.00001;
  if(r>=0.99999) r=0.99999;
  return mu+sigma*SqrtTwo*TMath::ErfcInverse(2.*(1.-r));  
}

//more scaling functions are available in priors.f90 in MultiNest package




