/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GStore                                                                *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <iostream>
#include <iomanip>
#include <algorithm>

#include "TMatrixD.h"

#include "Gfitter/GStore.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GFitterBase.h"
#include "Gfitter/GTheoryFactory.h"
#include "Gfitter/GAction.h"
#include "TFile.h"

using std::cout;
using std::endl;

Gfitter::GStore* Gfitter::gStore() 
{ 
   return Gfitter::GStore::Instance(); 
}

Gfitter::GStore* Gfitter::GStore::m_instance = 0;

Gfitter::GStore* Gfitter::GStore::Instance()
{
   if (m_instance == 0) m_instance = new GStore();
   return m_instance;
}

Gfitter::GStore* Gfitter::GStore::InitialInstance()
{
   return m_instance;
}

Gfitter::GStore::GStore()
   : m_correlationIsActive( kFALSE ),
     m_correlationMatrix( 0 ),
     m_covarianceMatrix( 0 ),
     m_fitter( 0 ),
     m_prefitter( 0 ),
     m_targetRootFile( 0 ),
     m_penaltyFactor( 0.15 ),
     m_msgLevel( kINFO )
{
   InitClassName( "GStore" );

   m_parameters.clear();
   m_activeParameters.clear();
   m_variables.clear();
   m_correlations.clear();
   m_actions.clear();
   m_activeUncorrelatedPars.clear();
   m_activeCorrelatedPars.clear();
}

Gfitter::GStore::~GStore()
{
   // need to du full cleanup !
   this->ClearStore();
}

void Gfitter::GStore::ClearStore()
{
   // delete all actions
   std::vector< const GAction* >::iterator iter = m_actions.begin();
   for (; iter!=m_actions.begin(); ++iter) {
     const GAction* action = *iter;
     m_actions.erase(iter);
     delete action;
   }
   m_actions.clear();

   // clear parameters and variables
   this->ClearParameters();
   // MB: and this clears the theories
   GTheoryFactory::Reset();

   // delete fitters
   if (m_fitter!=0)    { delete m_fitter; m_fitter=0; }
   if (m_prefitter!=0) { delete m_prefitter; m_prefitter=0; }

   // target ROOT file
   if (m_targetRootFile!=0) { m_targetRootFile->Close(); m_targetRootFile=0; }
}

void Gfitter::GStore::ClearParameters()
{
   // delete parameters and variables
   for (GParPtrVec_t::iterator parIt = m_parameters.begin(); 
        parIt != m_parameters.end(); parIt++) { delete (*parIt); (*parIt) = 0; }

   for (std::map<const TString, const GVariable*>::iterator varIt = m_variables.begin(); 
        varIt != m_variables.end(); varIt++) { delete (*varIt).second; (*varIt).second = 0; }
   m_variables.clear();

   m_parameters.clear();
   m_activeParameters.clear();
   m_correlations.clear();
   m_activeUncorrelatedPars.clear();
   m_activeCorrelatedPars.clear();

   if (m_correlationMatrix!=0) { delete m_correlationMatrix; m_correlationMatrix=0; }
   if (m_covarianceMatrix!=0)  { delete m_covarianceMatrix;  m_covarianceMatrix=0; }
}

UInt_t Gfitter::GStore::GetNumOfActiveParametersWithoutTheory() const
{
   UInt_t counter=0;
   GParPtrVec_t::const_iterator par = GetParameters().begin();
   for (; par != GetParameters().end(); par++) {
      if (!(*par)->HasTheory() && (*par)->IsActive()) counter++; 
   }
   return counter;
}
   
Gfitter::GParameter* Gfitter::GStore::GetParameter( const TString& name ) const
{
   // slow search; if this is critical would need hash table
   GParPtrVec_t::const_iterator par = GetParameters().begin();
   for (; par != GetParameters().end(); par++) {
      if ((*par)->GetParName() == name) return (*par);
   }
   return 0;      
}

Gfitter::GParameter* Gfitter::GStore::GetActiveParameter( const TString& name ) const
{
   // slow search; if this is critical would need hash table
   GParPtrVec_t::const_iterator par = GetParameters().begin();
   for (; par != GetParameters().end(); par++) {
      if ((*par)->IsActive() && (*par)->GetParName() == name) return (*par);
   }
   return 0;      
}

Gfitter::GParameter* Gfitter::GStore::GetParameterFromAlias( const TString& alias ) const
{
   // slow search; if this is critical would need hash table
   GParPtrVec_t::const_iterator par = GetParameters().begin();
   for (; par != GetParameters().end(); par++) {
      if ((*par)->GetParAlias() == alias) return (*par);
   }
   return 0;      
}

const Gfitter::GParPtrVec_t& Gfitter::GStore::GetActiveParameters()  
{ 
   m_activeParameters.clear();
   GParPtrVec_t::const_iterator par = GetParameters().begin();
   for (; par != GetParameters().end(); par++) {
      if ((*par)->IsActive()) m_activeParameters.push_back( *par );
   }
      
   return m_activeParameters; 
}

const Gfitter::GParPtrVec_t& Gfitter::GStore::GetActiveCorrelatedParameters() 
{
   return m_activeCorrelatedPars;      
}

const Gfitter::GParPtrVec_t& Gfitter::GStore::GetActiveUncorrelatedParameters()
{
   m_activeUncorrelatedPars.clear();
   const GParPtrVec_t& v = GetActiveParameters();
   for (GParPtrVec_t::const_iterator par = v.begin(); par != v.end(); par++) {
      if (std::find( m_activeCorrelatedPars.begin(), m_activeCorrelatedPars.end(), *par ) == 
          m_activeCorrelatedPars.end()) 
         m_activeUncorrelatedPars.push_back( *par );
   }
      
   return m_activeUncorrelatedPars;
}

void Gfitter::GStore::AddParameter( GParameter* gpar ) 
{ 
   if (gpar->IsActive()) m_activeParameters.push_back(gpar);
   m_parameters.push_back(gpar);

   // ----- on-the-fly updates of inter-parameter links

   // check for duplication, and link (active) duplicated parameters together
   const GParPtrVec_t& va = GetActiveParameters();
   for (GParPtrVec_t::const_iterator par = va.begin(); par != va.end(); par++) {
      if ((*par)->LinkIfSameParameter( gpar )) {
         // if linked, the test parameter cannot be first anymore
         gpar->UnsetFirst();
         m_logger << kINFO << "Parameter \"" << gpar->GetFullName() << "\" == \""
                  << (*par)->GetFullName() << "\": link these together (\"IsFirst\": " 
                  <<  gpar->IsFirst() << ", " << (*par)->IsFirst() << ")"
                  << GEndl;
      }
   }

   // check for parameter Scalers and init parameter links
   GParPtrVec_t::const_iterator par = va.begin();
   for (; par != va.end(); par++) (*par)->InitAllScalers( m_parameters );
}

void Gfitter::GStore::InitAllParameters()
{
   // ---- nothing to do for parameters

   // ---- final initialisation of all scalers 
   GParPtrVec_t::const_iterator par = GetParameters().begin();
   for (; par != GetParameters().end(); par++) {
      if ((*par)->HasScalers()) {

         GSclPtrVec_t::const_iterator scl = (*par)->GetScalers().begin();
         for (; scl != (*par)->GetScalers().end(); scl++) {
            
            // if a parameter already exist, skip it
            if ((*scl)->GetParameter()) continue;

            TString     expression = (*scl)->GetExpression();
            TString     maxName    = "";
            GParameter* maxPar     = 0;

            // loop again over all parameters and match to scaler
            GParPtrVec_t::const_iterator testpar = GetParameters().begin();            
            for (; testpar != GetParameters().end(); testpar++) {

               // parameter found in expression (find the largewst one that fits)
               TString parName = (*testpar)->GetFullName();
               if (expression.Contains( parName )) {
                  // replace if best name
                  if (parName.Sizeof() > maxName.Sizeof()) {
                     maxName = parName;
                     maxPar  = *testpar;
                  }
               }
            }

            // sanity check (should have been found)
            if (maxPar) (*scl)->SetParameter( maxPar );
            else m_logger << kFATAL << "Could not find parameter contained in scaler expression: \"" 
                          << expression << "\"" << GEndl;
         }         
      }
   }
   //   exit(1);
}

Double_t Gfitter::GStore::GetCorrelation( GParameter* gpar1, GParameter* gpar2 ) 
{
   return (m_correlations.find( this->GetCorrName( gpar1, gpar2 ) ) == m_correlations.end() ?
           0 : m_correlations[this->GetCorrName( gpar1, gpar2 )]);
}

void Gfitter::GStore::AddCorrelation( const TString& alias1, const TString& alias2, Double_t rho )
{ 
   // sanity check
   GParameter* gpar1 = this->GetParameterFromAlias( alias1 );      
   GParameter* gpar2 = this->GetParameterFromAlias( alias2 );
   if (gpar1 == 0 || gpar2 == 0) {
      m_logger << kFATAL << "<AddCorrelation> Requested correlation "
               << "for at least one unknown parameter aliases: " 
               << alias1 << " OR " << alias2 << " ==> abort" << GEndl;
   }
      
   this->AddCorrelation( gpar1, gpar2, rho );
}

void Gfitter::GStore::AddCorrelation( GParameter* gpar1, GParameter* gpar2, Double_t rho )
{ 
   // sanity checks
   if (gpar1 == 0 || gpar2 == 0) {
      m_logger << kFATAL << "<AddCorrelation> Zero GParameters: "
               << gpar1 << " OR " << gpar2 << " ==> abort" << GEndl;
   }

   // should not be two same parameters
   if (gpar1 == gpar2) {
      m_logger << kFATAL << "<AddCorrelation> Parameters are identical: "
               << gpar1->GetFullName() << " == " << gpar2->GetFullName() << " ==> abort" << GEndl;
   }

   // correlation for parameter pair shold not yet exist
   if (m_correlations.find( this->GetCorrName( gpar1, gpar2 ) ) != m_correlations.end()) {
      m_logger << kFATAL << "<AddCorrelations> Correlation for parameter pair "
               << gpar1->GetFullName() << " & " << gpar2->GetFullName() 
               << " already registered: " << m_correlations[GetCorrName( gpar1, gpar2 )] << GEndl;
   }

   // set the correlation pair
   m_correlations[this->GetCorrName( gpar1, gpar2 )] = rho;

   // add parameters to correlated parameter vector, if not yet present and if active 
   if (gpar1->IsActive() && 
       std::find( m_activeCorrelatedPars.begin(), m_activeCorrelatedPars.end(), gpar1 ) == m_activeCorrelatedPars.end())
      m_activeCorrelatedPars.push_back( gpar1 );
   if (gpar2->IsActive() && 
       std::find( m_activeCorrelatedPars.begin(), m_activeCorrelatedPars.end(), gpar2 ) == m_activeCorrelatedPars.end())
      m_activeCorrelatedPars.push_back( gpar2 );
}

const TString Gfitter::GStore::GetCorrName( const GParameter* gpar1, const GParameter* gpar2 ) const
{
   if (gpar1 == 0 || gpar2 == 0) m_logger << kFATAL << "<AddCorrelations> Zero GParameters: "
                                          << gpar1 << " OR " << gpar2 << GEndl;

   return gpar1->GetParAlias() + TString(":") + gpar2->GetParAlias();
}   

TMatrixD* Gfitter::GStore::GetCorrelationMatrix( const GParPtrVec_t& v )
{
   // delete if already existing
   if (m_correlationMatrix != 0) { delete m_correlationMatrix; m_correlationMatrix = 0; }

   if (!IsCorrelationActive()) return 0;

   Int_t n = v.size();

   // sanity check
   for (Int_t i = 0; i < n; i++) {
      if (v[i] == 0) m_logger << kFATAL << "<GetCorrelationMatrix> Zero pointer" << GEndl;
   }

   // create the matrix
   m_correlationMatrix = new TMatrixD( n, n );
   for (Int_t i = 0; i < n; i++) { 
      (*m_correlationMatrix)(i,i) = 1;
      for (Int_t j = i+1; j < n; j++) {
         (*m_correlationMatrix)(i,j) = GetCorrelation( v[i], v[j] );
         (*m_correlationMatrix)(j,i) = (*m_correlationMatrix)(i,j);
      }
   }

   return m_correlationMatrix;
}

TMatrixD* Gfitter::GStore::GetCovarianceMatrix( const GParPtrVec_t& v )
{
   m_correlationMatrix = GetCorrelationMatrix( v );
   if (m_correlationMatrix == 0) return 0;

   // delete if already existing
   if (m_covarianceMatrix != 0) { delete m_covarianceMatrix; m_covarianceMatrix = 0; }
      
   Int_t n = v.size();
   m_covarianceMatrix = new TMatrixD( n, n );
   
   // create the matrix
   for (Int_t i = 0; i < n; i++) { 

      // only for GData so far
      Gfitter::GData* tmpData = dynamic_cast<Gfitter::GData*>(v[i]->GetDataPtr());
      if (tmpData !=0 ) { 
         
         Double_t erri = tmpData->GetErrGaussSym();         
         
         // sanity check
         if (erri <= 0) {
            m_logger << kFATAL << "<GetCovarianceMatrix> erri: " << erri << " <= 0 in parameter: " 
                     << v[i]->GetFullName() << GEndl;
         }
         
         // at present, only covariance matrices for symmetric errors are
         // implemented; publish a warning if asymmetric errors are present
         if (tmpData->GetErrGaussp() != tmpData->GetErrGaussm()) {
            m_logger << kWARNING << "<GetCovarianceMatrix> Parameter: \""
                     << v[i]->GetFullName() 
                     << "\": asymmetric errors together with correlations are not yet supported "
                     << " ==> symmetrize errors" << GEndl;
            
         }// diagonal elements
         (*m_covarianceMatrix)(i,i) = erri*erri;
         
         // off-diagonal elements
         for (Int_t j = i+1; j < n; j++) {

            // only for GData so far
            Gfitter::GData* tmpData2 = dynamic_cast<Gfitter::GData*>(v[j]->GetDataPtr());
            if (tmpData2 !=0 ){
               Double_t errj = tmpData2->GetErrGaussSym();
               if (errj <= 0) {
                  m_logger << kFATAL << "<GetCovarianceMatrix> " 
                           << "errj: " << errj << " <= 0 in parameter: " 
                           << v[j]->GetFullName() << GEndl;
               }
               (*m_covarianceMatrix)(i,j) = (*m_correlationMatrix)(i,j)*erri*errj;
               (*m_covarianceMatrix)(j,i) = (*m_covarianceMatrix)(i,j);
            }
         }
      }
   }
   
   return m_covarianceMatrix;
}

void Gfitter::GStore::AddAction( const GAction* action )
{ 
   if (!action) m_logger << kFATAL << "<AddAction> Cannot add zero pointer to action list" << GEndl;
   m_actions.push_back( action );
}

Bool_t Gfitter::GStore::ExistVariable( const TString& key )
{
   return (m_variables[key] != 0) ? kTRUE : kFALSE;
}

const Gfitter::GVariable* Gfitter::GStore::GetVariable( const TString& key ) 
{
   // slow search; if this is critical would need hash table
   // don't allow to retrieve unknown object
   const GVariable* f = m_variables[key];
   if (f == 0) {
      m_logger << kFATAL << "<GetVariable> Variable for key : \"" << key 
               << "\" does not exist" << GEndl;
   }
   return f;
}

const Gfitter::GVariable* Gfitter::GStore::GetVariableIfExist( const TString& key ) 
{
   // slow search; if this is critical would need hash table
   return m_variables[key];
}

void Gfitter::GStore::AddVariable( const GVariable* variable ) 
{ 
   if (variable == 0) m_logger << kFATAL << "<AddVariable> Cannot add zero pointer"  << GEndl;

   m_variables[variable->GetVariableName()] = variable;

   // some variables have member status
   if (variable->GetVariableName() == "GfitterFlags::OutputLevel") {
      // sanity check
      if (variable->GetType() != GVariable::String) {
         m_logger << kFATAL << "<AddVariable> Variable \"OutputLevel\" "
                  << "must be of type string, but is: " << variable->GetType() << GEndl;
      }
      SetMsgLevel( m_logger.MapLevel( variable->GetStringValue() ) );

      // update all loggers that were initialised so far
      GMsgLogger::SetMinLevel( GetMsgLevel() );

      m_logger << kINFO << "Set minimum message logging level to: \"" 
               << m_logger.GetMinLevelStr() << "\"" << GEndl;
   }
}

void Gfitter::GStore::AddVariable( TString key, TString value ) 
{ 
   this->AddVariable( new const GVariable( key, value ) );
}

void Gfitter::GStore::SetVariable( TString key, TString value )
{
  if ( this->ExistVariable(key) ) { 
    std::map<const TString, const GVariable*>::iterator itr = m_variables.find(key);
    delete itr->second;
    m_variables.erase(itr);
  }
  this->AddVariable(key,value);
}

Gfitter::GParameter* Gfitter::GStore::SearchParameterInString( const TString& parName ) const
{
   for (Int_t length=1; length<parName.Sizeof(); length++) {
      for (Int_t index=0; index<parName.Sizeof()-length; index++) {
         TString substr = parName(index,length);
         GParameter* par = GetParameter( substr );
         if (par != 0) return par;
      }
   }

   return 0;
}

void Gfitter::GStore::BackupParameters()
{
   Gfitter::GParPtrVec_t::const_iterator it = GetParameters().begin();
   for (; it != GetParameters().end(); it++) (*it)->Backup();   
}

void Gfitter::GStore::RecoverParameters()
{
   Gfitter::GParPtrVec_t::const_iterator it = GetParameters().begin();
   for (; it != GetParameters().end(); it++) (*it)->Recover();   
}

void Gfitter::GStore::DumpParameters()
{
   // dump all active parameters, and compare to associated theory (if exists)

   // get largest parameter width
   Int_t maxL = 10;
   Int_t maxS = 7;
   Gfitter::GParPtrVec_t::const_iterator it;
   for (it = GetActiveParameters().begin(); it != GetActiveParameters().end(); it++) {
      maxL = TMath::Max(maxL, (Int_t)((*it)->GetFullName().Sizeof()));
      if ((*it)->HasScalers()) maxS = TMath::Max(maxS, (Int_t)((*it)->GetScalerStr().Sizeof()));
   }

   Int_t   length = 64 + 9 + maxL + maxS + 3;
   TString line; for (Int_t i=0; i<length; i++) line += "-";

   // print
   m_logger << kINFO << GEndl;
   m_logger << kINFO << line << GEndl;
   m_logger << kINFO << "G S t o r e   A c t i v e   P a r a m e t e r   D u m p " << GEndl;
   m_logger << kINFO << line << GEndl;
   using std::setw;
   using std::left;
   using std::right;
   TString valFormat = "%11.7g";
   TString errFormat = "%11.7g";
   TString rngFormat = "%7.3g";
   m_logger << kINFO << left
            << setw(maxL-1) << "Parameter" << "    "
            << setw(13) << "    Value" 
            << setw(13) << " FitValue"
            << setw(26) << "             Gauss error"
            << setw(19) << "       Range error " << "|"
            << setw(11) << "     Theory" << " | "
            << right
            << setw(maxS) << "Scalers"
            << left
            << GEndl;
   m_logger << kINFO << line << GEndl;
   for (it = GetActiveParameters().begin(); it != GetActiveParameters().end(); it++) {
      GData* tmpData = dynamic_cast<GData*>( (*it)->GetDataPtr() );      
      
      if (tmpData) {
         m_logger << kINFO 
                  << left
                  << setw(maxL-1) << (*it)->GetFullName() << " ="
                  << setw(9) << Form( valFormat, (*it)->GetValue() ) << " " << Form( valFormat, (*it)->GetFitValue() ) 
                  << setw(3) << "  +" << setw(7) << Form( errFormat, tmpData->GetErrGaussp() ) 
                  << setw(3) << "  -" << setw(7) << Form( errFormat, tmpData->GetErrGaussm() ) 
                  << setw(3) << "  +" << setw(5) << Form( rngFormat, tmpData->GetErrRangep() ) 
                  << setw(3) << "  -" << setw(5) << Form( rngFormat, tmpData->GetErrRangem() ) 
                  << " |"
                  << right
                  << setw(11) << ((*it)->HasTheory() ? 
                                 Form( valFormat, (*it)->GetTheoryPrediction() ) : "--")
                  << setw(3) << " | "
                  << setw(maxS) << ((*it)->HasScalers() ? (*it)->GetScalerStr() : "--")
                  << left
                  << GEndl;
      }
      else {
         m_logger << kINFO << "Cannot display parameter: \"" << (*it)->GetFullName() 
                  << "\" which has a data object of type GDataCurve or GWorkspace" << GEndl;
      }      
   }
   m_logger << kINFO << line << GEndl;
   m_logger << GEndl;
}

ostream& Gfitter::operator << ( ostream& o, const Gfitter::GStore& s ) 
{
   o << "GStore contains the following paramters:" << std::endl
     << "----------------------------------------" << std::endl;
   for (Gfitter::GParPtrVec_t::const_iterator parIt = s.GetParameters().begin(); 
        parIt != s.GetParameters().end(); parIt++) o << *(*parIt) << std::endl;

   o << "GStore contains the following variables:" << std::endl
     << "----------------------------------------" << std::endl;      
   for (std::map<const TString, const Gfitter::GVariable*>::const_iterator varIt = s.GetVariables().begin(); 
        varIt != s.GetVariables().end(); varIt++) o << *(*varIt).second << std::endl;

   return o;
}
