/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GTheoryBase                                                           *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/

#include <iostream>
#include "Gfitter/GTheoryFactory.h"
#include "Gfitter/GTheoryBase.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GParameterRef.h"

ClassImp(Gfitter::GTheoryBase)

Gfitter::GTheoryBase::GTheoryBase() 
   : Gfitter::GObject(),
     m_theoryName( "None" ),
     m_instanceName( "None" ),
     m_existDerivative( kFALSE ),
     m_useParCaching( kFALSE ),
     m_isUpToDate( kFALSE )
{
   GObject::InitClassName( GetName() );
   
   m_TmpParBroker.clear();

   if (gStore()->ExistVariable( "GfitterFlags::UseParCaching" )) {
      m_useParCaching = gStore()->GetVariable( "GfitterFlags::UseParCaching" )->GetBoolValue();
   }
}
   
// constructor for simple formula parser
Gfitter::GTheoryBase::GTheoryBase( const TString &name, const TString& /* formula */ )
   : Gfitter::GObject(),
     m_theoryName( name ),
     m_instanceName( "None" ),
     m_existDerivative( kFALSE ),
     m_useParCaching( kFALSE ),
     m_isUpToDate( kFALSE )
{
   GObject::InitClassName( GetName() );
   
   // formula parser... not yet implemented (use TF1)
   m_logger << kFATAL << "Construction from formula not yet implemented" << GEndl;
   m_TmpParBroker.clear();

   if (gStore()->ExistVariable( "GfitterFlags::UseParCaching" )) {
      m_useParCaching = gStore()->GetVariable( "GfitterFlags::UseParCaching" )->GetBoolValue();
   }
}
   
Gfitter::GTheoryBase::~GTheoryBase() {}

Bool_t Gfitter::GTheoryBase::InitParameters() 
{
   Bool_t success = kTRUE;
   std::map<const TString, GParameterRef*>::const_iterator parIt = m_TmpParBroker.begin();
   for (; parIt!=m_TmpParBroker.end(); parIt++) {

      GParameter* gpar = gStore()->GetActiveParameter( parIt->first );
      
      // check whether an active parameter exists
      if (gpar == 0) {

         m_logger << kINFO << "<InitParameters> Parameter \"" << parIt->first << "\" is not active" << GEndl;
         m_logger << kINFO << "<InitParameters> check also deactivated parameters" << GEndl;

         // otherwise use deactived parameters
         gpar = gStore()->GetParameter( parIt->first );
         
         //ceck whether an parameter is registered at all
         if (gpar == 0) 
            m_logger << kFATAL << "<InitParameters> Unknown parameter: \"" << parIt->first << "\"" << GEndl;
      }

      GParameterRef * parRef = parIt->second;
      
      parRef->SetOwner(this);
      
      parRef->SetAddress( gpar );
      
      // register theory with the parameter
      // the parameter then knows that this theory is using it
      m_logger << kINFO << "<InitTheories> RegisterClientReference of p: \"" << gpar->GetFullName() << "\"" << GEndl;
      gpar->RegisterClientReference(parRef);
      
   }
   return success;
}

Bool_t Gfitter::GTheoryBase::InitTheories() 
{

   // loop over all booked theories
   Bool_t success = kTRUE;
   std::map<const TString, GTheoryRef*>::const_iterator thIt = m_TmpTheoryBroker.begin();
   for (; thIt!=m_TmpTheoryBroker.end(); thIt++) {
      GTheoryBase* theory = GTheoryFactory::GetTheory( thIt->first );

      // check whether theory has been registered
      if (theory == 0) {
         m_logger << kFATAL << "<InitTheories> Unknown theory: \"" << thIt->first << "\"" << GEndl;
         m_logger << "<InitTheories> Problem with requested theory. " << GEndl;
         m_logger << "<InitTheories> Here is the list of all theories created so far: " << GEndl;
         std::map<const TString, GTheoryBase*>::iterator it = GTheoryFactory::CreatedTheories().begin();
         for (; it != GTheoryFactory::CreatedTheories().end(); it++) {
            m_logger << kINFO << "   " << it->first << " --> " << it->second->GetClassName() << GEndl;
         }
      }
               
      GTheoryRef * thRef = thIt->second;
      
      thRef->SetOwner(this);
      thRef->SetAddress( theory );
      
      // register this theory with the theories that it uses, this
      // is done by telling them about the reference this theory
      // has on them
      m_logger << kINFO << "<InitTheories> RegisterClientRefenrece of t: \"" << thIt->first << "\"" << GEndl;
      theory->RegisterClientReference(thRef);
   }
   return success;
}

void Gfitter::GTheoryBase::BookParameter( const TString& parname, GParameterRef* par ) 
{
   m_TmpParBroker[parname] = par;
}

void Gfitter::GTheoryBase::BookTheory( const TString& parname, GTheoryRef* theory ) 
{
   m_TmpTheoryBroker[parname] = theory;
}

ostream& Gfitter::GTheoryBase::Cout( ostream& os, int indLevel, int counter ) const
{
   for(Int_t i=0; i<3*indLevel; i++) os << " ";
   os << indLevel << "." << counter << "   " << this->GetTheoryName() << std::endl;
   for(Int_t i=0; i<3*indLevel; i++) os << " ";
   os << "   used theories ["<< m_TmpTheoryBroker.size() <<"]:" << std::endl;
   std::map<const TString,GTheoryRef*>::const_iterator thIt = this->m_TmpTheoryBroker.begin();
   int idx = 1;
   for(;thIt!=m_TmpTheoryBroker.end(); thIt++) {
      GTheoryRef & ref     = *thIt->second;
      ref->Cout(os,indLevel+1, idx++);
   }
   
   for(Int_t i=0; i<3*indLevel; i++) os << " ";
   os << "used parameters ["<< m_TmpParBroker.size() <<"]:" << std::endl;
   std::map<const TString,GParameterRef*>::const_iterator parIt = this->m_TmpParBroker.begin();
   for(;parIt!=m_TmpParBroker.end(); parIt++) {
      const TString & parName = parIt->first;
      //			GParameterRef * ref     = parIt->Second;
      for(Int_t i=0; i<3*(indLevel+1); i++) os << " ";
      os << parName << std::endl;
   }

   return os;
}

void Gfitter::GTheoryBase::RegisterClientReference( GTheoryRef* th ) 
{
   if ( m_clientReferences.FindObject(th) != 0 ) {
      m_logger << kWARNING << "The theory \"" 
               << GetTheoryName() << "\" already has the theory \""
               << (*th)->GetTheoryName() << "\" as a client" << GEndl;
      return;
   }
   m_clientReferences.Add(th);
}
   
void Gfitter::GTheoryBase::NotifyReferencesOfChange() 
{
   TListIter clIt( &m_clientReferences );
   while (GTheoryRef* theoryRef = (GTheoryRef*)clIt()) { 
      //      m_logger << "SetChanged " << theoryRef->GetName() << GEndl; 
      theoryRef->SetChanged();
   }
}

void Gfitter::GTheoryBase::BeNotified(GReference & ref) 
{
   //   m_logger << kINFO << GetTheoryName() << "::BeNotified()   " << "UptoDate: " << IsUpToDate()<< GEndl;
   if (IsUpToDate() || this->InheritsFrom( "Gfitter::GAuxTheory" ) ) {     
      //      m_logger << "xx Updating local flags: " << GEndl;
      UpdateLocalFlags(ref);
      //      m_logger << "xx Updating uptodate flags: " << GEndl;
      m_isUpToDate = kFALSE;
      //      m_logger << "xx BeNotified: Entering: NotifyReferencesOfChange() " << GetTheoryName() << GEndl;
      NotifyReferencesOfChange();
   }
}

void Gfitter::GTheoryBase::SetReferenceUsed(GReference * ref) 
{
   if (m_isUpToDate == kTRUE) return;
   std::map<const TString,GParameterRef*>::iterator parRefIt = m_TmpParBroker.begin();
   std::map<const TString,GTheoryRef*>::iterator thRefIt = m_TmpTheoryBroker.begin();
   m_isUpToDate = kTRUE;
   if (ref==0) {
      // set all to "used", meaning the theory itself is uptodate
      for (; parRefIt != m_TmpParBroker.end(); parRefIt++) parRefIt->second->SetUsed();
      for (; thRefIt != m_TmpTheoryBroker.end(); thRefIt++) thRefIt->second->SetUsed();
   }
   else {
      ref->SetUsed();
      for (;parRefIt != m_TmpParBroker.end() && m_isUpToDate;parRefIt++)
         m_isUpToDate &= ! parRefIt->second->HasChanged();
      for (;thRefIt != m_TmpTheoryBroker.end() && m_isUpToDate;thRefIt++)
         m_isUpToDate &= ! thRefIt->second->HasChanged();
   }
}

ostream& Gfitter::operator<< ( ostream& os, const Gfitter::GTheoryBase& theory ) 
{   
   return theory.Cout( os );
}
