/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GTimer                                                                *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 *      Adapted from TMVA:Timer. Original authors:                                *
 *                                                                                *                 
 *      Andreas Hoecker <Andreas.Hocker@cern.ch> - CERN, Switzerland              *
 *      Joerg Stelzer   <Joerg.Stelzer@cern.ch>  - CERN, Switzerland              * 
 *      Helge Voss      <Helge.Voss@cern.ch>     - MPI-K Heidelberg, Germany      * 
 *      Kai Voss        <Kai.Voss@cern.ch>       - U. of Victoria, Canada         *
 *                                                                                *
 *                                                                                *
 * Copyright (c) 2006                                                             *
 *      CERN, Switzerland                                                         *
 *      MPI-K Heidelberg, Germany                                                 *
 *                                                                                *
 *   http://root.cern.ch/root/htmldoc/src/TMVA__Timer.h.html                      *
 *                                                                                *
 * (http://tmva.sourceforge.net/LICENSE)                                          *           
 **********************************************************************************/

//_______________________________________________________________________
//                                                                      
// Timing information for training and evaluation of MVA methods  
// 
// Usage:
//
//    Gfitter::GTimer timer( Nloops, "MyClassName" ); 
//    for (Int_t i=0; i<Nloops; i++) {
//      ... // some code
//
//      // now, print progress bar:
//      timer.DrawProgressBar( i );
//
//      // **OR** text output of left time (never both !)
//      m_logger << " time left: " << timer.GetLeftTime( i ) << Endl;
//
//    } 
//    m_logger << "MyClassName" << ": elapsed time: " << timer.GetElapsedTime() 
//            << Endl;    
//
// Remark: in batch mode, the progress bar is quite ugly; you may 
//         want to use the text output then
//_______________________________________________________________________

#include <cstdlib>
#include "Gfitter/GTimer.h"
#include "Riostream.h"

const TString Gfitter::GTimer::m_gClassName = "GTimer";
const Int_t   Gfitter::GTimer::fgNbins     = 24;  

//_______________________________________________________________________
Gfitter::GTimer::GTimer( const char* prefix, Bool_t colourfulOutput )
   : m_ncounts        ( 0 ),
     m_prefix         ( prefix ),
     m_colourfulOutput( colourfulOutput )
{
   // constructor
   if (m_prefix=="") m_prefix = Gfitter::GTimer::m_gClassName;

   m_logger = new GMsgLogger( m_prefix.Data() );

   Reset();
}

//_______________________________________________________________________
Gfitter::GTimer::GTimer( Int_t ncounts, const char* prefix, Bool_t colourfulOutput  )
   : m_ncounts        ( ncounts ),
     m_prefix         ( prefix ),
     m_colourfulOutput( colourfulOutput )
{
   // standard constructor: ncounts gives the total number of counts that 
   // the loop will iterate through. At each call of the timer, the current
   // number of counts is provided by the user, so that the timer can obtain
   // the due time from linearly interpolating the spent time.

   if (m_prefix=="") m_prefix = Gfitter::GTimer::m_gClassName;

   m_logger = new GMsgLogger( m_prefix.Data() );

   Reset();
}

//_______________________________________________________________________
Gfitter::GTimer::~GTimer( void )
{
   // destructor
   delete m_logger;
}

void Gfitter::GTimer::Init( Int_t ncounts )
{
   // timer initialisation
   m_ncounts = ncounts;  
   Reset();
}

//_______________________________________________________________________
void Gfitter::GTimer::Reset( void )
{
   // resets timer
   TStopwatch::Start( kTRUE );
}

//_______________________________________________________________________
Double_t Gfitter::GTimer::ElapsedSeconds( void ) 
{
   // computes elapsed tim in seconds
   Double_t rt = TStopwatch::RealTime(); TStopwatch::Start( kFALSE );
   return rt;
}
//_______________________________________________________________________

TString Gfitter::GTimer::GetElapsedTime( Bool_t Scientific ) 
{
   // returns pretty string with elaplsed time
   return SecToText( ElapsedSeconds(), Scientific );
}

//_______________________________________________________________________
TString Gfitter::GTimer::GetLeftTime( Int_t icounts ) 
{
   // returns pretty string with time left
   Double_t leftTime = ( icounts <= 0 ? -1 :
                         icounts > m_ncounts ? -1 :
                         Double_t(m_ncounts - icounts)/Double_t(icounts)*ElapsedSeconds() );

   return SecToText( leftTime, kFALSE );
}

//_______________________________________________________________________
void Gfitter::GTimer::DrawProgressBar() 
{
   // draws the progressbar
   m_ncounts++;
   if (m_ncounts == 1) {
      clog << m_logger->GetPrintedSource();
      clog << "Please wait ";
   }

   clog << "." << flush;
}

//_______________________________________________________________________
void Gfitter::GTimer::DrawProgressBar( TString theString ) 
{
   // draws a string in the progress bar
   clog << m_logger->GetPrintedSource();

   clog << Color("white_on_green") << Color("dyellow") << "[" << Color("reset");

   clog << Color("white_on_green") << Color("dyellow") << theString << Color("reset");

   clog << Color("white_on_green") << Color("dyellow") << "]" << Color("reset") 
        << "                  ";

   clog << "\r" << flush; 
}

//_______________________________________________________________________
void Gfitter::GTimer::DrawProgressBar( Int_t icounts ) 
{
   // draws progress bar in color or B&W
   // caution: 

   // don't draw each time...
   if (icounts > 0 && m_ncounts > 0 && icounts%Int_t(m_ncounts/100.0) != 0) return;

   // sanity check:
   if (icounts > m_ncounts-1) icounts = m_ncounts-1;
   if (icounts < 0          ) icounts = 0;
   Int_t ic = Int_t(Float_t(icounts)/Float_t(m_ncounts)*fgNbins);

   clog << m_logger->GetPrintedSource();
   if (m_colourfulOutput) clog << Color("white_on_green") << Color("dyellow") << "[" << Color("reset");
   else                  clog << "[";
   for (Int_t i=0; i<ic; i++) {
      if (m_colourfulOutput) clog << Color("white_on_green") << Color("dyellow") << ">" << Color("reset"); 
      else                  clog << ">";
   }
   for (Int_t i=ic+1; i<fgNbins; i++) {
      if (m_colourfulOutput) clog << Color("white_on_green") << Color("dyellow") << "." << Color("reset"); 
      else                  clog << ".";
   }
   if (m_colourfulOutput) clog << Color("white_on_green") << Color("dyellow") << "]" << Color("reset");
   else                  clog << "]" ;

   // timing information
   if (m_colourfulOutput) {
      clog << Color("reset") << " " ;
      clog << "(" << Color("red") << Int_t((100*(icounts+1))/Float_t(m_ncounts)) << "%" << Color("reset")
               << ", " 
               << "time left: "
               << this->GetLeftTime( icounts ) << Color("reset") << ") ";
   }
   else {
      clog << "] " ;
      clog << "(" << Int_t((100*(icounts+1))/Float_t(m_ncounts)) << "%" 
               << ", " << "time left: " << this->GetLeftTime( icounts ) << ") ";
   }
   clog << "\r" << flush; 
}

//_______________________________________________________________________
TString Gfitter::GTimer::SecToText( Double_t seconds, Bool_t Scientific ) const
{
   // pretty string output
   TString out = "";
   if      (Scientific    ) out = Form( "%.3g sec", seconds );
   else if (seconds <  0  ) out = "unknown";
   else if (seconds <= 300) out = Form( "%i sec", Int_t(seconds) );
   else {
      if (seconds > 3600) {
         Int_t h = Int_t(seconds/3600);
         if (h <= 1) out = Form( "%i hr : ", h );
         else        out = Form( "%i hrs : ", h );
      
         seconds = Int_t(seconds)%3600;
      }
      Int_t m = Int_t(seconds/60);
      if (m <= 1) out += Form( "%i min", m );
      else        out += Form( "%i mins", m );
   }

   return (m_colourfulOutput) ? Color("red") + out + Color("reset") : out;
}

//_______________________________________________________________________
const TString& Gfitter::GTimer::Color( TString c ) const
{
   // human readable color strings
   static TString gClr_none         = "" ;
   static TString gClr_white        = "\033[1;37m";  // white
   static TString gClr_blue         = "\033[34m";    // blue
   static TString gClr_red          = "\033[1;31m" ; // red
   static TString gClr_yellow       = "\033[1;33m";  // yellow
   static TString gClr_darkred      = "\033[31m";    // dark red
   static TString gClr_darkgreen    = "\033[32m";    // dark green
   static TString gClr_darkyellow   = "\033[33m";    // dark yellow
                                    
   static TString gClr_white_b      = "\033[1m"    ; // bold white
   static TString gClr_lblue_b      = "\033[1;34m" ; // bold light blue
   static TString gClr_lgreen_b     = "\033[1;32m";  // bold light green
                                    
   static TString gClr_blue_bg      = "\033[44m";    // blue background
   static TString gClr_red_bg       = "\033[1;41m";  // white on red background
   static TString gClr_whiteonblue  = "\033[1;44m";  // white on blue background
   static TString gClr_whiteongreen = "\033[1;42m";  // white on green background
   static TString gClr_grey_bg      = "\033[47m";    // grey background

   static TString gClr_reset  = "\033[0m";     // reset

   if (c=="white")   return gClr_white; 
   if (c=="blue")    return gClr_blue; 
   if (c=="yellow")  return gClr_yellow; 
   if (c=="red")     return gClr_red; 
   if (c=="dred")    return gClr_darkred; 
   if (c=="dgreen")  return gClr_darkgreen; 
   if (c=="dyellow") return gClr_darkyellow; 

   if (c=="bwhite")  return gClr_white_b; 

   if (c=="blue_bgd") return gClr_blue_bg; 
   if (c=="red_bgd" ) return gClr_red_bg; 
 
   if (c=="white_on_blue" )  return gClr_whiteonblue; 
   if (c=="white_on_green")  return gClr_whiteongreen; 

   if (c=="reset") return gClr_reset; 

   cout << "Unknown color " << c << endl;
   exit(1);

   return gClr_none;
}
