/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GToyAnalysis                                                          *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <iostream>
#include "TMatrixD.h"
#include "TArrayD.h"
#include "TTree.h"
#include "TMath.h"
#include "TAxis.h"
#include "TH1F.h"
#include "TH2F.h"

#include "Gfitter/GToyAnalysis.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GCorrGaussGen.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GFitterBase.h"
#include "Gfitter/GMinuitFitter.h"
#include "Gfitter/GTimer.h"
#include "Gfitter/GInterval.h"

using namespace std;

Gfitter::GToyAnalysis::GToyAnalysis()
{
   InitClassName( "GToyAnalysis" );
}

Gfitter::GToyAnalysis::~GToyAnalysis()
{}

Double_t Gfitter::GToyAnalysis::ExecuteToy( Double_t initialChi2min, Int_t nToyExp  ) 
{
   Int_t NumOfExpHigherChi2 = 0 ;

   m_logger << kINFO << "Starting toy analysis with " << nToyExp 
            << " experiments (initialChi2min = " << initialChi2min << ")" << GEndl; 

   // book parameter tree
   TTree* tree = new TTree( "toy_result", "Results of toy test" );

   // get already fitted parameters
   GParPtrVec_t corrFitPars   = gStore()->GetActiveCorrelatedParameters();
   GParPtrVec_t uncorrFitPars = gStore()->GetActiveUncorrelatedParameters();

   // and backup the fit values
   TArrayD corrParamFitValues  ( corrFitPars.size() );
   TArrayD uncorrParamFitValues( uncorrFitPars.size() );

   m_logger << kDEBUG << "Number of correlated parameters: " << corrFitPars.size()<< GEndl;
   for (UInt_t ipar=0; ipar<corrFitPars.size(); ipar++) {

      // correlated randomising not implemented for GDataCurve!
      if (corrFitPars[ipar]->IsCurve())
         m_logger << kFATAL << "Correlated randomising not implemented for GDataCurves"<< GEndl;

      if (!corrFitPars[ipar]->HasTheory()) {
         m_logger << kDEBUG << "Parameter: " << corrFitPars[ipar]->GetParName() 
                  << " value: " << corrFitPars[ipar]->GetValue() 
                  << " fit value: " << corrFitPars[ipar]->GetFitValue() 
                  << " errorm: " << corrFitPars[ipar]->GetErrGaussm()
                  << " errorp: " << corrFitPars[ipar]->GetErrGaussp() << GEndl;

         if (corrFitPars[ipar]->HasScalers()) {
            const GSclPtrVec_t& scalers = corrFitPars[ipar]->GetScalers();
            for (unsigned int i=0; i<scalers.size(); i++) {
               scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
            }
         }
         corrParamFitValues[ipar] = corrFitPars[ipar]->GetFitValue();
      }
      else {
         m_logger << kDEBUG << "Parameter: " << corrFitPars[ipar]->GetParName() 
                  << " value: " << corrFitPars[ipar]->GetValue() 
                  << " fit value: " << corrFitPars[ipar]->GetTheoryPrediction()
                  << " errorm: " << corrFitPars[ipar]->GetErrGaussm()
                  << " errorp: " << corrFitPars[ipar]->GetErrGaussp() << GEndl;

         if (corrFitPars[ipar]->HasScalers()) {
            const GSclPtrVec_t& scalers = corrFitPars[ipar]->GetScalers();
            for (unsigned int i=0; i<scalers.size(); i++) {
               scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
            }
         }
         corrParamFitValues[ipar] = corrFitPars[ipar]->GetTheoryPrediction();
      }
   }
   
   m_logger << kDEBUG << "Number of uncorrelated parameters: " << uncorrFitPars.size()<< GEndl;
   for (UInt_t ipar=0; ipar<uncorrFitPars.size(); ipar++) {
      if (!uncorrFitPars[ipar]->HasTheory()) {
         m_logger << kDEBUG << "Parameter: " << uncorrFitPars[ipar]->GetParName() 
                  << " value: " << uncorrFitPars[ipar]->GetValue() 
                  << " fit value: " << uncorrFitPars[ipar]->GetFitValue()
                  << " errorm: " << uncorrFitPars[ipar]->GetErrGaussm()
                  << " errorp: " << uncorrFitPars[ipar]->GetErrGaussp() << GEndl;
         
         if (uncorrFitPars[ipar]->HasScalers()) {
            const GSclPtrVec_t& scalers = uncorrFitPars[ipar]->GetScalers();
            for (unsigned int i=0; i<scalers.size(); i++) {
               scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
            }
         }

         uncorrParamFitValues[ipar] = uncorrFitPars[ipar]->GetFitValue();
      }
      else {
         m_logger << kDEBUG << "Parameter: " << uncorrFitPars[ipar]->GetParName() 
                  << " value: " << uncorrFitPars[ipar]->GetValue() 
                  << " fit value: " << uncorrFitPars[ipar]->GetTheoryPrediction()
                  << " errorm: " << uncorrFitPars[ipar]->GetErrGaussm()
                  << " errorp: " << uncorrFitPars[ipar]->GetErrGaussp() << GEndl;
                     
         if (uncorrFitPars[ipar]->HasScalers()) {
            const GSclPtrVec_t& scalers = uncorrFitPars[ipar]->GetScalers();
            for (unsigned int i=0; i<scalers.size(); i++) {
               scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
            }
         }

         uncorrParamFitValues[ipar] = uncorrFitPars[ipar]->GetTheoryPrediction();
      }
   }
 
   // sanity check
   if (uncorrFitPars.size()+corrFitPars.size() != gStore()->GetActiveParameters().size() )
      m_logger << kFATAL 
               << "Number of active parameters not equal to sum of correlated and uncorrelated" 
               << GEndl;

   // create the tree branches
   Double_t x_rndval[uncorrFitPars.size()+corrFitPars.size()];
   Double_t x_fitval[uncorrFitPars.size()+corrFitPars.size()];
   Double_t x_chi2tot, x_chi2tot_ini;
   Int_t i = 0;
   for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
        par1 != gStore()->GetActiveParameters().end(); par1++) {
      TString n = Form( "rnd_%s", (*par1)->GetFullNameWONSpace().Data() );
      tree->Branch( n, &x_rndval[i], GUtils::ReplaceRegularExpressions( n, '_' ) + "/D" );

      n = Form( "fit_%s", (*par1)->GetFullNameWONSpace().Data() );
      tree->Branch( n, &x_fitval[i], GUtils::ReplaceRegularExpressions( n, '_' ) + "/D" );
      i++;     
   }
   tree->Branch("chi2_tot" , &x_chi2tot, "chi2_tot/D" );
   tree->Branch("chi2_tot_ini" , &x_chi2tot_ini, "chi2_tot_ini/D" );

   // vectors to store the random numbers
   TArrayD corrRndParamVector  ( corrFitPars.size() );
   TArrayD uncorrRndParamVector( uncorrFitPars.size() );

   // get the covariance matrix 
   TMatrixD* covMatrix = gStore()->GetCovarianceMatrix( corrFitPars   );

   // initialize randomising of correlated parameters
   GCorrGaussGen corrGaussGen( covMatrix );

   // another sanity check: test that the new "measurements" (those created from the perfect theory)
   // have a chi2 = 0
   for (Int_t ipar=0; ipar<corrRndParamVector.GetSize(); ipar++) {
      if  (corrFitPars[ipar]->GetErrGaussm()!=0 &&
           corrFitPars[ipar]->GetErrGaussp()!=0 ) corrFitPars[ipar]->SetValue( corrParamFitValues[ipar] );
   }
   for (Int_t ipar=0; ipar<uncorrRndParamVector.GetSize(); ipar++) {
      if  (uncorrFitPars[ipar]->GetErrGaussm()!=0 &&
           uncorrFitPars[ipar]->GetErrGaussp()!=0 )
         uncorrFitPars[ipar]->SetValue( uncorrParamFitValues[ipar] );
   }

   // refit
   Double_t chi2test = gStore()->GetFitter()->GetCurrentChi2();
   m_logger << kINFO << "Test chi2 after toy set creation (should be zero): " << chi2test << GEndl;
   if (chi2test > 1e-05) m_logger << kWARNING << "<Execute>: chi2test = " << chi2test << " > 0" << GEndl;
   
   // timing                  
   Int_t iloop = 0;
   GTimer timer( nToyExp, GetPrintName() );

   // loop over toy experiments
   for (Int_t itoy=0; itoy<nToyExp; itoy++) {  
      
      // get random numbers for correlated parameters that have random error (ie. measurement) and set it 
      corrGaussGen.GetGaussRnd( corrRndParamVector );
    
      for (Int_t ipar=0; ipar<corrRndParamVector.GetSize(); ipar++) {
         if  (corrFitPars[ipar]->GetErrGaussm()!=0 &&
              corrFitPars[ipar]->GetErrGaussp()!=0 ) {
            Double_t newvalue = corrRndParamVector[ipar] + corrParamFitValues[ipar];            
            m_logger << kDEBUG << "Corr parameter: " << corrFitPars[ipar]->GetParName() 
                     << " old value : " << corrParamFitValues[ipar]  
                     << " new value : " << newvalue << GEndl;
            corrFitPars[ipar]->SetValue( newvalue );
         }
      }
      
      // get random numbers for uncorrelated parameters that have random error (ie. measurement) and set it
      for (Int_t ipar=0; ipar<uncorrRndParamVector.GetSize(); ipar++) {
         // is not used for Direct Higgs Searches
         //          if ( uncorrFitPars[ipar]->IsCurve() ) {
         //             Double_t newvalue = uncorrFitPars[ipar]->GetRndValue(uncorrParamFitValues[ipar]);
         //             m_logger << kINFO << "Uncorr and curve parameter: " << uncorrFitPars[ipar]->GetParName() 
         //                      << " old value " << uncorrParamFitValues[ipar]  
         //                      << " new value " << newvalue << GEndl;
         //             //uncorrFitPars[ipar]->SetValue( newvalue );
         //          }
         if( uncorrFitPars[ipar]->GetErrGaussm()!=0 &&
             uncorrFitPars[ipar]->GetErrGaussp()!=0 ) {
            Double_t newvalue = uncorrFitPars[ipar]->GetGaussRnd(uncorrParamFitValues[ipar]); 
            m_logger << kDEBUG << "Uncorr parameter: " << uncorrFitPars[ipar]->GetParName() 
                     << " old value " << uncorrParamFitValues[ipar]  
                     << " new value " << newvalue << GEndl;
            uncorrFitPars[ipar]->SetValue( newvalue );
         }
      }
      // refit
      Double_t chi2min=0.;
      chi2min = gStore()->GetFitter()->ExecuteFit();  

      m_logger << kINFO
               << Form( "Experiment:%4i | chi2: %7.4g | time left: %s",
                        itoy, chi2min, timer.GetLeftTime( ++iloop ).Data() )
               << GEndl;

      // count number of experiments with a chi2 higher than the initial fit
      if (chi2min>initialChi2min) NumOfExpHigherChi2++;
      
      if(chi2min < 1){
      // some debug output
      for (GParPtrVec_t::const_iterator par2 = gStore()->GetActiveParameters().begin(); 
           par2 != gStore()->GetActiveParameters().end(); par2++) {
         if (!(*par2)->HasTheory()){
            m_logger << kINFO << "After fit : " << (*par2)->GetFullNameWONSpace()
                     << " value : " << (*par2)->GetFitValue() << GEndl;
            m_logger << kINFO << "chi2 = " << (*par2)->GetChiSquared() << GEndl;}
         else{
            m_logger << kINFO << "After fit : " << (*par2)->GetFullNameWONSpace() 
                     << " value : " << (*par2)->GetTheoryPrediction() << GEndl;
            m_logger << kINFO << "chi2 = " << (*par2)->GetChiSquared() << GEndl;}
      }

      }
      // fill the tree
      Int_t k=0;
      for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
           par1 != gStore()->GetActiveParameters().end(); par1++) {
         x_rndval[k]  = (*par1)->GetValue();
         if (!(*par1)->HasTheory())
            x_fitval[k]  = (*par1)->GetFitValue();
         else 
            x_fitval[k]  = (*par1)->GetTheoryPrediction();
         k++; 
      }
      x_chi2tot = chi2min;
      x_chi2tot_ini = initialChi2min;

      tree->Fill();      
   }
   
   m_logger << kINFO << "Elapsed time: " << timer.GetElapsedTime() 
            << "                                      " << GEndl;    

   // write the tree
   tree    ->Write();
   delete tree;
   
   // calculate the p-value
   Double_t pvalue  = Double_t(NumOfExpHigherChi2)/Double_t(nToyExp);
   Double_t epvalue = TMath::Sqrt( pvalue*(1.0 - pvalue)/Double_t(nToyExp) );

   m_logger << kINFO << "P-value after ToyTest: " << pvalue << " +- " << epvalue << GEndl ;
  
   return pvalue;
}

//----------------- one-dimensional Toy Scan --------------------------

void Gfitter::GToyAnalysis::Scan1DToy( const TString& par, Int_t nbins, Int_t ntoys )
{
   GParameter* gpar = gStore()->GetParameter( par );
  
   // sanity checks
   if (gpar == 0) m_logger << kFATAL << "<Scan1DToy> Fatal error: unknown parameter \"" << par << "\": "
                            << " check datacard" << GEndl;
   
   // parameter must be active
   if (!gpar->IsActive())
      m_logger << kFATAL << "<Scan1DToy> Fatal error: Scanned Parameter must be active in Toy Scan, check datacard" << GEndl;

   // not tested for any kind of measurements!!!
   if( (gpar->GetErrGaussm() == 0 && gpar->GetErrGaussp() == 0) || gpar->IsCurve() )
      m_logger << kWARNING << "<Scan1DToy>: Not tested or parameters with any kind of measurement" << GEndl;
      
   this->Scan1DToy( gpar, nbins, ntoys );
}

void  Gfitter::GToyAnalysis::Scan1DToy( GParameter* gpar, Int_t nbins, Int_t ntoys ) 
{
   // sanity check
   if (gpar == 0 ) m_logger << kFATAL << "<Scan1DToy> Zero GParameter pointer: " << gpar << GEndl;

   m_logger << kINFO << "Scanning parameter \"" << gpar->GetParName() << "\""
            << " (full name: \"" << gpar->GetFullName() << "\")"
            << " in interval [" << gpar->GetScanRange().GetMin() << ", " << gpar->GetScanRange().GetMax() << "]"
            << GEndl;
   m_logger << kINFO << "Number of steps in scan: " << nbins << GEndl;

   // backup paramter settings
   gpar->Backup();
  
   // create histograms (also serves as scan axis)
   TString bulkName = gpar->GetFullNameWONSpace();
   
   TH1F* th1CL    = new TH1F( bulkName + (TString)"_cl", 
                              bulkName + (TString)" (1 - CL)", 
                              nbins, gpar->GetScanRange().GetMin(), gpar->GetScanRange().GetMax() );
   
   // store values from data card
   GParPtrVec_t ActivePars   = gStore()->GetActiveParameters();
   
   // and backup the values (from datacard)
   TArrayD ParamValues  ( ActivePars.size() );
   
   for (UInt_t ipar=0; ipar<ActivePars.size(); ipar++) {
      ParamValues[ipar] = ActivePars[ipar]->GetValue();
   }
   
   //gStore()->BackupParameters(); needs to be debugged
   
   // initialise free parameters in fitter 
   gStore()->GetFitter()->Initialise();

   // compute intial chi2 with free scan parameter
   Double_t inichi2free = gStore()->GetFitter()->ExecuteFit( );

   
   m_logger << kINFO << "Scanning ... please be patient" << GEndl;
   
   for (Int_t ibin1=1; ibin1<=th1CL->GetNbinsX(); ++ibin1) {
      
      // get stored values
      for (UInt_t ipar=0; ipar<ActivePars.size(); ipar++) {
         ActivePars[ipar]->SetValue( ParamValues[ipar]);
         ActivePars[ipar]->SetFitValue( ParamValues[ipar]);
      }
      //gStore()->RecoverParameters(); needs to be debugged

      // set parameter on scan value
      gpar->SetValue( th1CL->GetXaxis()->GetBinCenter( ibin1 ) );
      gpar->SetFitValue( th1CL->GetXaxis()->GetBinCenter( ibin1 ) );
      
      // parameter inactive 
      gpar->SetActive( kFALSE );
      gStore()->GetFitter()->Initialise();
      
      // compute chi2 with fixed parameter 
      Double_t inichi2     = gStore()->GetFitter()->ExecuteFit( );

      // get already fitted parameters
      GParPtrVec_t corrFitPars   = gStore()->GetActiveCorrelatedParameters();
      GParPtrVec_t uncorrFitPars = gStore()->GetActiveUncorrelatedParameters();
      
      // and backup the fit values
      TArrayD corrParamFitValues  ( corrFitPars.size() );
      TArrayD uncorrParamFitValues( uncorrFitPars.size() );
      
      m_logger << kDEBUG << "Number of correlated parameters: " << corrFitPars.size()<< GEndl;
      for (UInt_t ipar=0; ipar<corrFitPars.size(); ipar++) {
         
         // correlated randomising not implemented for GDataCurve!
         if (corrFitPars[ipar]->IsCurve())
            m_logger << kFATAL << "Correlated randomising not implemented for GDataCurves"<< GEndl;
         
         if (!corrFitPars[ipar]->HasTheory()) {
            m_logger << kDEBUG << "Parameter: " << corrFitPars[ipar]->GetParName() 
                     << " value: " << corrFitPars[ipar]->GetValue() 
                     << " fit value: " << corrFitPars[ipar]->GetFitValue() 
                     << " errorm: " << corrFitPars[ipar]->GetErrGaussm()
                     << " errorp: " << corrFitPars[ipar]->GetErrGaussp() << GEndl;
            
            if (corrFitPars[ipar]->HasScalers()) {
               const GSclPtrVec_t& scalers = corrFitPars[ipar]->GetScalers();
               for (unsigned int i=0; i<scalers.size(); i++) {
                  scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
               }
            }
            corrParamFitValues[ipar] = corrFitPars[ipar]->GetFitValue();
         }
         else {
            m_logger << kDEBUG << "Parameter: " << corrFitPars[ipar]->GetParName() 
                     << " value: " << corrFitPars[ipar]->GetValue() 
                     << " fit value: " << corrFitPars[ipar]->GetTheoryPrediction()
                     << " errorm: " << corrFitPars[ipar]->GetErrGaussm()
                     << " errorp: " << corrFitPars[ipar]->GetErrGaussp() << GEndl;
            
            if (corrFitPars[ipar]->HasScalers()) {
               const GSclPtrVec_t& scalers = corrFitPars[ipar]->GetScalers();
               for (unsigned int i=0; i<scalers.size(); i++) {
                  scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
               }
            }
            corrParamFitValues[ipar] = corrFitPars[ipar]->GetTheoryPrediction();
         }
      }
         
      m_logger << kDEBUG << "Number of uncorrelated parameters: " << uncorrFitPars.size()<< GEndl;
      for (UInt_t ipar=0; ipar<uncorrFitPars.size(); ipar++) {
         if (!uncorrFitPars[ipar]->HasTheory()) {
            m_logger << kDEBUG << "Parameter: " << uncorrFitPars[ipar]->GetParName() 
                     << " value: " << uncorrFitPars[ipar]->GetValue() 
                     << " fit value: " << uncorrFitPars[ipar]->GetFitValue()
                     << " errorm: " << uncorrFitPars[ipar]->GetErrGaussm()
                     << " errorp: " << uncorrFitPars[ipar]->GetErrGaussp() << GEndl;
            
            if (uncorrFitPars[ipar]->HasScalers()) {
               const GSclPtrVec_t& scalers = uncorrFitPars[ipar]->GetScalers();
               for (unsigned int i=0; i<scalers.size(); i++) {
                  scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
               }
            }
            
            uncorrParamFitValues[ipar] = uncorrFitPars[ipar]->GetFitValue();
         }
         else {
            m_logger << kDEBUG << "Parameter: " << uncorrFitPars[ipar]->GetParName() 
                     << " value: " << uncorrFitPars[ipar]->GetValue() 
                     << " fit value: " << uncorrFitPars[ipar]->GetTheoryPrediction()
                     << " errorm: " << uncorrFitPars[ipar]->GetErrGaussm()
                     << " errorp: " << uncorrFitPars[ipar]->GetErrGaussp() << GEndl;
            
            if (uncorrFitPars[ipar]->HasScalers()) {
               const GSclPtrVec_t& scalers = uncorrFitPars[ipar]->GetScalers();
               for (unsigned int i=0; i<scalers.size(); i++) {
                  scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
               }
            }
               
            uncorrParamFitValues[ipar] = uncorrFitPars[ipar]->GetTheoryPrediction();
         }
      }
         
      // vectors to store the random numbers
      TArrayD corrRndParamVector  ( corrFitPars.size() );
      TArrayD uncorrRndParamVector( uncorrFitPars.size() );
         
      // get the covariance matrix 
      TMatrixD* covMatrix = gStore()->GetCovarianceMatrix( corrFitPars   );

      // initialize randomising of correlated parameters
      GCorrGaussGen corrGaussGen( covMatrix );
         
      // sanity check
      if (uncorrFitPars.size()+corrFitPars.size() != gStore()->GetActiveParameters().size() )
         m_logger << kFATAL 
                  << "Number of active parameters not equal to sum of correlated and uncorrelated" 
                  << GEndl;
   
      // another sanity check: test that the new "measurements" (those created from the perfect theory)
      // have a chi2 = 0
      for (Int_t ipar=0; ipar<corrRndParamVector.GetSize(); ipar++) {
         if  (corrFitPars[ipar]->GetErrGaussm()!=0 &&
              corrFitPars[ipar]->GetErrGaussp()!=0 ) corrFitPars[ipar]->SetValue( corrParamFitValues[ipar] );
      }
      for (Int_t ipar=0; ipar<uncorrRndParamVector.GetSize(); ipar++) {
         if  (uncorrFitPars[ipar]->GetErrGaussm()!=0 &&
              uncorrFitPars[ipar]->GetErrGaussp()!=0 )
            uncorrFitPars[ipar]->SetValue( uncorrParamFitValues[ipar] );
      }
      
      // refit
      Double_t chi2test = gStore()->GetFitter()-> GetCurrentChi2();
      if (chi2test > 1e-05) m_logger << kWARNING << "<Execute>: chi2test = " << chi2test << " > 0" << GEndl;
         
                  
      Int_t NoOfToysLargerChi2 = 0;
         
      // toy scan
      for ( Int_t itoy = 0; itoy < ntoys; itoy++ ){
                       
         // re-set paramter on scan value
         // this is done because the last fit in this loop with free scan parameter
         // changes the value of paramter again
         gpar->SetValue( th1CL->GetXaxis()->GetBinCenter( ibin1 ) );
         gpar->SetFitValue( th1CL->GetXaxis()->GetBinCenter( ibin1 ) );

         // get random numbers for correlated parameters that have random error (ie. measurement) and set it 
         corrGaussGen.GetGaussRnd( corrRndParamVector );
         
         for (Int_t ipar=0; ipar<corrRndParamVector.GetSize(); ipar++) {
            if  (corrFitPars[ipar]->GetErrGaussm()!=0 &&
                 corrFitPars[ipar]->GetErrGaussp()!=0 ) {
               Double_t newvalue = corrRndParamVector[ipar] + corrParamFitValues[ipar];            
               m_logger << kDEBUG << "Corr parameter: " << corrFitPars[ipar]->GetParName() 
                        << " old value : " << corrParamFitValues[ipar]  
                        << " new value : " << newvalue << GEndl;
               corrFitPars[ipar]->SetValue( newvalue );
               corrFitPars[ipar]->SetFitValue( newvalue );
            }
         }
            
         // get random numbers for uncorrelated parameters that have random error (ie. measurement) and set it
         for (Int_t ipar=0; ipar<uncorrRndParamVector.GetSize(); ipar++) {
            // is not used for Direct Higgs Searches
            //          if ( uncorrFitPars[ipar]->IsCurve() ) {
            //             Double_t newvalue = uncorrFitPars[ipar]->GetRndValue(uncorrParamFitValues[ipar]);
            //             m_logger << kINFO << "Uncorr and curve parameter: " << uncorrFitPars[ipar]->GetParName() 
            //                      << " old value " << uncorrParamFitValues[ipar]  
            //                      << " new value " << newvalue << GEndl;
            //             //uncorrFitPars[ipar]->SetValue( newvalue );
            //          }
            if( uncorrFitPars[ipar]->GetErrGaussm()!=0 &&
                uncorrFitPars[ipar]->GetErrGaussp()!=0 ) {
               Double_t newvalue = uncorrFitPars[ipar]->GetGaussRnd(uncorrParamFitValues[ipar]); 
               m_logger << kDEBUG << "Uncorr parameter: " << uncorrFitPars[ipar]->GetParName() 
                        << " old value " << uncorrParamFitValues[ipar]  
                        << " new value " << newvalue << GEndl;
               uncorrFitPars[ipar]->SetValue( newvalue );
               uncorrFitPars[ipar]->SetFitValue( newvalue );
            }
         }
                 
         // compute chi2 with fixed parameter
         Double_t chi2     = gStore()->GetFitter()->ExecuteFit( );   

         // parameter active 
         gpar->SetActive( kTRUE );
         gStore()->GetFitter()->Initialise();
         
         // compute chi2 with free parameter
         Double_t chi2free = gStore()->GetFitter()->ExecuteFit( );
      
         // parameter inactive 
         gpar->SetActive( kFALSE );
         gStore()->GetFitter()->Initialise();

         if ( chi2 - chi2free < -10e-5 ){
            m_logger << kFATAL << "<Scan1DToy> " << chi2 << " < " << chi2free 
                     << ": chi2 with free scan parameter must be smaller than the chi2 computed with fixed scan parameter" << GEndl;
         }

         // Delta_chi2_toy > Delta_chi2_ini
         if ( chi2 - chi2free > inichi2 - inichi2free ) NoOfToysLargerChi2++;
         
      }
         
      // compute CL
      Double_t cl  = (Double_t)NoOfToysLargerChi2/(Double_t)ntoys;
      Double_t err = TMath::Sqrt( cl*(1-cl)/(Double_t)ntoys );
           
      // fill the histogram
      th1CL->SetBinContent( ibin1, cl );
      th1CL->SetBinError  ( ibin1, err );
   }
   
   // save histogram  
   th1CL   ->Write();
   
   // recover initial settings
   gpar->Recover();
}

//----------------- two-dimensional Toy Scan --------------------------

void Gfitter::GToyAnalysis::Scan2DToy( const TString& par1, const TString& par2, Int_t nbinsX, Int_t nbinsY, Int_t ntoys )
{
   GParameter* gpar1 = gStore()->GetParameter( par1 );
   GParameter* gpar2 = gStore()->GetParameter( par2 );
  
   // sanity checks
   if (gpar1 == 0 ) m_logger << kFATAL << "<Scan2DToy> Fatal error: unknown parameter \"" << par1 << "\": "
                             << " check datacard" << GEndl;
   if (gpar2 == 0 ) m_logger << kFATAL << "<Scan2DToy> Fatal error: unknown parameter \"" << par2 << "\": "
                             << " check datacard" << GEndl;

   // parameter must be active
   if (!gpar1->IsActive() || !gpar2->IsActive() )
      m_logger << kFATAL << "<Scan2DToy> Fatal error: Scanned Parameters must be active in Toy Scan, check datacard" << GEndl;

   // not tested for any kind of measurements!!!
   if( (gpar1->GetErrGaussm() != 0 || gpar1->GetErrGaussp() != 0) || gpar1->IsCurve() )
      m_logger << kWARNING << "<Scan2DToy>: Not tested for parameters with any kind of measurement: " 
               << par1 << GEndl;
   if( (gpar2->GetErrGaussm() != 0 || gpar2->GetErrGaussp() != 0) || gpar2->IsCurve() )
      m_logger << kWARNING << "<Scan2DToy>: Not tested for parameters with any kind of measurement: "
               << par2 << GEndl;
      
   this->Scan2DToy( gpar1, gpar2, nbinsX, nbinsY, ntoys );
}

void  Gfitter::GToyAnalysis::Scan2DToy( GParameter* gpar1, GParameter* gpar2, Int_t nbinsX, Int_t nbinsY, Int_t ntoys ) 
{
   // sanity check
   if (gpar1 == 0 || gpar2 == 0) m_logger << kFATAL << "<Scan2D> Zero GParameter pointer: " 
                                          << gpar1 << " or " << gpar2 << GEndl;

   m_logger << kINFO << "Scanning parameters \"" << gpar2->GetParName() 
            << "\" versus \"" << gpar1->GetParName() << "\"" 
            << " in the intervals [" << gpar2->GetScanRange().GetMin() << ", " 
            << gpar2->GetScanRange().GetMax() << "] and ["<< gpar1->GetScanRange().GetMin() << ", " 
            << gpar1->GetScanRange().GetMax() << "]"
            << GEndl;
   m_logger << kINFO << "Number of steps in scan X-axis: " << nbinsX << GEndl;
   m_logger << kINFO << "Number of steps in scan Y-axis: " << nbinsY << GEndl;

   // backup paramter settings
   gpar1->Backup();
   gpar2->Backup();
  
   // create histograms (also serves as scan axis)
   TString bulkName = gpar2->GetParNameWONSpace() + (TString)"_vs_" + gpar1->GetParNameWONSpace();
   
   TH2F* th2CL    = new TH2F( bulkName + (TString)"_cl", 
                              bulkName + (TString)" (1 - CL)", 
                              nbinsX, gpar1->GetScanRange().GetMin(), gpar1->GetScanRange().GetMax(),
                              nbinsY, gpar2->GetScanRange().GetMin(), gpar2->GetScanRange().GetMax() );

   // store values from data card
   GParPtrVec_t ActivePars   = gStore()->GetActiveParameters();
         
   // and backup the values (from datacard)
   TArrayD ParamValues  ( ActivePars.size() );
   
   for (UInt_t ipar=0; ipar<ActivePars.size(); ipar++) {
      ParamValues[ipar] = ActivePars[ipar]->GetValue();
   }

   //gStore()->BackupParameters(); needs to be debugged

   // initialise free parameters in fitter 
   gStore()->GetFitter()->Initialise();

   // compute intial chi2 with free scan parameter
   Double_t inichi2free = gStore()->GetFitter()->ExecuteFit( );

   
   m_logger << kINFO << "Scanning ... please be patient" << GEndl;
   
   for (Int_t ibin1=1; ibin1<=th2CL->GetNbinsX(); ++ibin1) {
      for (Int_t ibin2=1; ibin2<=th2CL->GetNbinsY(); ++ibin2) {
         
         // get stored values
         for (UInt_t ipar=0; ipar<ActivePars.size(); ipar++) {
            ActivePars[ipar]->SetValue( ParamValues[ipar]);
            ActivePars[ipar]->SetFitValue( ParamValues[ipar]);
         }
         //gStore()->RecoverParameters(); needs to be debugged

         // set parameters 
         gpar1->SetValue( th2CL->GetXaxis()->GetBinCenter( ibin1 ) );
         gpar1->SetFitValue( th2CL->GetXaxis()->GetBinCenter( ibin1 ) );
         gpar2->SetValue( th2CL->GetYaxis()->GetBinCenter( ibin2 ) );
         gpar2->SetFitValue( th2CL->GetYaxis()->GetBinCenter( ibin2 ) );
      
         // parameter inactive 
         gpar1->SetActive( kFALSE );
         gpar2->SetActive( kFALSE );
         gStore()->GetFitter()->Initialise();
      
         // compute chi2 with fixed parameter 
         Double_t inichi2     = gStore()->GetFitter()->ExecuteFit( );
         Double_t pVal_1 = TMath::Prob( inichi2 - inichi2free, 1);
         Double_t pVal_2 = TMath::Prob( inichi2 - inichi2free, 2);
         
         Double_t cl;
         if( pVal_1 > 0.66 )
            cl = pVal_1;
         else if ( pVal_2 < 0.005 )
            cl = pVal_2;
         else {

            // get already fitted parameters
            GParPtrVec_t corrFitPars   = gStore()->GetActiveCorrelatedParameters();
            GParPtrVec_t uncorrFitPars = gStore()->GetActiveUncorrelatedParameters();
      
            // and backup the fit values
            TArrayD corrParamFitValues  ( corrFitPars.size() );
            TArrayD uncorrParamFitValues( uncorrFitPars.size() );
      
            m_logger << kDEBUG << "Number of correlated parameters: " << corrFitPars.size()<< GEndl;
            for (UInt_t ipar=0; ipar<corrFitPars.size(); ipar++) {
         
               // correlated randomising not implemented for GDataCurve!
               if (corrFitPars[ipar]->IsCurve())
                  m_logger << kFATAL << "Correlated randomising not implemented for GDataCurves"<< GEndl;
         
               if (!corrFitPars[ipar]->HasTheory()) {
                  m_logger << kDEBUG << "Parameter: " << corrFitPars[ipar]->GetParName() 
                           << " value: " << corrFitPars[ipar]->GetValue() 
                           << " fit value: " << corrFitPars[ipar]->GetFitValue() 
                           << " errorm: " << corrFitPars[ipar]->GetErrGaussm()
                           << " errorp: " << corrFitPars[ipar]->GetErrGaussp() << GEndl;
            
                  if (corrFitPars[ipar]->HasScalers()) {
                     const GSclPtrVec_t& scalers = corrFitPars[ipar]->GetScalers();
                     for (unsigned int i=0; i<scalers.size(); i++) {
                        scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
                     }
                  }
                  corrParamFitValues[ipar] = corrFitPars[ipar]->GetFitValue();
               }
               else {
                  m_logger << kDEBUG << "Parameter: " << corrFitPars[ipar]->GetParName() 
                           << " value: " << corrFitPars[ipar]->GetValue() 
                           << " fit value: " << corrFitPars[ipar]->GetTheoryPrediction()
                           << " errorm: " << corrFitPars[ipar]->GetErrGaussm()
                           << " errorp: " << corrFitPars[ipar]->GetErrGaussp() << GEndl;
            
                  if (corrFitPars[ipar]->HasScalers()) {
                     const GSclPtrVec_t& scalers = corrFitPars[ipar]->GetScalers();
                     for (unsigned int i=0; i<scalers.size(); i++) {
                        scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
                     }
                  }
                  corrParamFitValues[ipar] = corrFitPars[ipar]->GetTheoryPrediction();
               }
            }
         
            m_logger << kDEBUG << "Number of uncorrelated parameters: " << uncorrFitPars.size()<< GEndl;
            for (UInt_t ipar=0; ipar<uncorrFitPars.size(); ipar++) {
               if (!uncorrFitPars[ipar]->HasTheory()) {
                  m_logger << kDEBUG << "Parameter: " << uncorrFitPars[ipar]->GetParName() 
                           << " value: " << uncorrFitPars[ipar]->GetValue() 
                           << " fit value: " << uncorrFitPars[ipar]->GetFitValue()
                           << " errorm: " << uncorrFitPars[ipar]->GetErrGaussm()
                           << " errorp: " << uncorrFitPars[ipar]->GetErrGaussp() << GEndl;
            
                  if (uncorrFitPars[ipar]->HasScalers()) {
                     const GSclPtrVec_t& scalers = uncorrFitPars[ipar]->GetScalers();
                     for (unsigned int i=0; i<scalers.size(); i++) {
                        scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
                     }
                  }
            
                  uncorrParamFitValues[ipar] = uncorrFitPars[ipar]->GetFitValue();
               }
               else {
                  m_logger << kDEBUG << "Parameter: " << uncorrFitPars[ipar]->GetParName() 
                           << " value: " << uncorrFitPars[ipar]->GetValue() 
                           << " fit value: " << uncorrFitPars[ipar]->GetTheoryPrediction()
                           << " errorm: " << uncorrFitPars[ipar]->GetErrGaussm()
                           << " errorp: " << uncorrFitPars[ipar]->GetErrGaussp() << GEndl;
            
                  if (uncorrFitPars[ipar]->HasScalers()) {
                     const GSclPtrVec_t& scalers = uncorrFitPars[ipar]->GetScalers();
                     for (unsigned int i=0; i<scalers.size(); i++) {
                        scalers[i]->SetRefValue( scalers[i]->GetParameter()->GetFitValue() );
                     }
                  }
               
                  uncorrParamFitValues[ipar] = uncorrFitPars[ipar]->GetTheoryPrediction();
               }
            }
         
            // vectors to store the random numbers
            TArrayD corrRndParamVector  ( corrFitPars.size() );
            TArrayD uncorrRndParamVector( uncorrFitPars.size() );
         
            // get the covariance matrix 
            TMatrixD* covMatrix = gStore()->GetCovarianceMatrix( corrFitPars   );

            // initialize randomising of correlated parameters
            GCorrGaussGen corrGaussGen( covMatrix );
         
            // sanity check
            if (uncorrFitPars.size()+corrFitPars.size() != gStore()->GetActiveParameters().size() )
               m_logger << kFATAL 
                        << "Number of active parameters not equal to sum of correlated and uncorrelated" 
                        << GEndl;
   
            // another sanity check: test that the new "measurements" (those created from the perfect theory)
            // have a chi2 = 0
            for (Int_t ipar=0; ipar<corrRndParamVector.GetSize(); ipar++) {
               if  (corrFitPars[ipar]->GetErrGaussm()!=0 &&
                    corrFitPars[ipar]->GetErrGaussp()!=0 ) corrFitPars[ipar]->SetValue( corrParamFitValues[ipar] );
            }
            for (Int_t ipar=0; ipar<uncorrRndParamVector.GetSize(); ipar++) {
               if  (uncorrFitPars[ipar]->GetErrGaussm()!=0 &&
                    uncorrFitPars[ipar]->GetErrGaussp()!=0 )
                  uncorrFitPars[ipar]->SetValue( uncorrParamFitValues[ipar] );
            }
      
            // refit
            Double_t chi2test = gStore()->GetFitter()-> GetCurrentChi2();
            if (chi2test > 1e-05) m_logger << kWARNING << "<Execute>: chi2test = " << chi2test << " > 0" << GEndl;
         
                  
            Int_t NoOfToysLargerChi2 = 0;
         
            // toy scan
            for ( Int_t itoy = 0; itoy < ntoys; itoy++ ){
                       
               // re-set paramter on scan value
               // this is done because the last fit in this loop with free scan parameter
               // changes the value of paramter again
               // set parameters on scan values
               gpar1->SetValue( th2CL->GetXaxis()->GetBinCenter( ibin1 ) );
               gpar1->SetFitValue( th2CL->GetXaxis()->GetBinCenter( ibin1 ) );
               gpar2->SetValue( th2CL->GetYaxis()->GetBinCenter( ibin2 ) );
               gpar2->SetFitValue( th2CL->GetYaxis()->GetBinCenter( ibin2 ) );
      
               // get random numbers for correlated parameters that have random error (ie. measurement) and set it 
               corrGaussGen.GetGaussRnd( corrRndParamVector );
         
               for (Int_t ipar=0; ipar<corrRndParamVector.GetSize(); ipar++) {
                  if  (corrFitPars[ipar]->GetErrGaussm()!=0 &&
                       corrFitPars[ipar]->GetErrGaussp()!=0 ) {
                     Double_t newvalue = corrRndParamVector[ipar] + corrParamFitValues[ipar];            
                     m_logger << kDEBUG << "Corr parameter: " << corrFitPars[ipar]->GetParName() 
                              << " old value : " << corrParamFitValues[ipar]  
                              << " new value : " << newvalue << GEndl;
                     corrFitPars[ipar]->SetValue( newvalue );
                     corrFitPars[ipar]->SetFitValue( newvalue );
                  }
               }
            
               // get random numbers for uncorrelated parameters that have random error (ie. measurement) and set it
               for (Int_t ipar=0; ipar<uncorrRndParamVector.GetSize(); ipar++) {
                  // is not used for Direct Higgs Searches
                  //          if ( uncorrFitPars[ipar]->IsCurve() ) {
                  //             Double_t newvalue = uncorrFitPars[ipar]->GetRndValue(uncorrParamFitValues[ipar]);
                  //             m_logger << kINFO << "Uncorr and curve parameter: " << uncorrFitPars[ipar]->GetParName() 
                  //                      << " old value " << uncorrParamFitValues[ipar]  
                  //                      << " new value " << newvalue << GEndl;
                  //             //uncorrFitPars[ipar]->SetValue( newvalue );
                  //          }
                  if( uncorrFitPars[ipar]->GetErrGaussm()!=0 &&
                      uncorrFitPars[ipar]->GetErrGaussp()!=0 ) {
                     Double_t newvalue = uncorrFitPars[ipar]->GetGaussRnd(uncorrParamFitValues[ipar]); 
                     m_logger << kDEBUG << "Uncorr parameter: " << uncorrFitPars[ipar]->GetParName() 
                              << " old value " << uncorrParamFitValues[ipar]  
                              << " new value " << newvalue << GEndl;
                     uncorrFitPars[ipar]->SetValue( newvalue );
                     uncorrFitPars[ipar]->SetFitValue( newvalue );
                  }
               }
         
               // compute chi2 with fixed parameter
               Double_t chi2     = gStore()->GetFitter()->ExecuteFit( );   

               // parameter inactive 
               gpar1->SetActive( kTRUE );
               gpar2->SetActive( kTRUE );
               gStore()->GetFitter()->Initialise();
         
               // compute chi2 with free parameter
               Double_t chi2free = gStore()->GetFitter()->ExecuteFit( );
            
               // parameter inactive 
               gpar1->SetActive( kFALSE );
               gpar2->SetActive( kFALSE );
               gStore()->GetFitter()->Initialise();
         
               if ( chi2 - chi2free < -10e-5 ){
                  m_logger << kFATAL << "<Scan2DToy>" << chi2 << " < " << chi2free 
                           << ": chi2 with free scan parameter must be smalle than the chi2 computed with fixed scan parameter" << GEndl;
               }

               // Delta_chi2_toy > Delta_chi2_ini
               if ( chi2 - chi2free > inichi2 - inichi2free ) NoOfToysLargerChi2++;
            
               m_logger << kINFO << gpar1->GetParName() << " vs. " << gpar2->GetParName() 
                        << ": ( " << gpar1->GetValue() << ", " << gpar2->GetValue() << " ): "
                        << itoy << " of " << ntoys << GEndl;
            }
         
            // compute CL
            cl  = (Double_t)NoOfToysLargerChi2/(Double_t)ntoys;

            m_logger << kINFO << "Finished Toy Analysis with " << ntoys << " experiments!" << GEndl;
         }
         
         Double_t err = TMath::Sqrt( cl*(1-cl)/(Double_t)ntoys );
         
         m_logger << kINFO << gpar1->GetParName() << " vs. " << gpar2->GetParName() 
                  << ": ( " << gpar1->GetValue() << ", " << gpar2->GetValue() << " ): Found p-value = "
                  << cl << " +- " << err << GEndl;

         // fill the histogram
         th2CL->SetBinContent( ibin1, ibin2, cl );
         th2CL->SetBinError  ( ibin1, ibin2, err );
      }
   }
   
   // save histogram  
   th2CL   ->Write();
   
   // recover initial settings
   gpar1->Recover();
   gpar2->Recover();
}

ostream& Gfitter::operator << ( ostream& os, const Gfitter::GToyAnalysis& /* toy */ ) 
{
   os << 0;
   return os;
}
